#include "Leggi.h"


/*
*
* ClipToBorders()
*
* This routine takes a Window pointer and creates and installs a clip inside
* the window borders. The old Region (for a pass to UnClipToBorder) is returned.
*
*/

struct Region *ClipToBorders(struct Window *Window) {

	struct Region *Region;
	struct Rectangle Rectangle;

	Region = NewRegion();
	Rectangle.MinX = Window->BorderLeft+1;
	Rectangle.MinY = Window->BorderTop+SKIPTOP;
	Rectangle.MaxX = Window->Width - Window->BorderRight - 1;
	Rectangle.MaxY = Window->Height - Window->BorderBottom - 1;
	OrRectRegion(Region, &Rectangle);
	return(InstallClipRegion(Window->WLayer, Region));
}

/*
*
* UnClipToBorders()
*
* This routine is the "destroy" routine for the ClipToBorders() "create" routine.
* It takes a Window pointer and an Region creates and installs the region,
* Dispose()ing of the current.
*
*/

void UnClipToBorders(struct Window *Window, struct Region *OldRegion) {

	struct Region *Region;

	if (Region = InstallClipRegion(Window->WLayer, OldRegion)) DisposeRegion(Region);
}


/*
*
* OpenSystemFont()
*
* takes a FontTable and puts in all of the entries (basing itself on
* MAXFONTS) the system font.
*
*/

void OpenSystemFont(struct TextFont **FontTable) {

	struct TextAttr TextAttr;
	int i;

	TextAttr.ta_Name = GfxBase->DefaultFont->tf_Message.mn_Node.ln_Name;
	TextAttr.ta_YSize = GfxBase->DefaultFont->tf_YSize;
	TextAttr.ta_Style = GfxBase->DefaultFont->tf_Style;
	TextAttr.ta_Flags = GfxBase->DefaultFont->tf_Flags;

	FontTable[MAXFONTS] = OpenFont(&TextAttr);
	for(i=0; i<MAXFONTS; i++) FontTable[i] = FontTable[MAXFONTS];

}

/*
*
* OpenFontNamed()
*
* takes a char *, a default height and tries to open that font. If no size
* is specified, calls the default height. NOTE: name is expected to have
* room for the ".info" suffix.
*
*/

struct TextFont *OpenFontNamed(char *name, int defheight) {

	struct TextAttr TextAttr;
	struct TextFont *Font;
	char *c=NULL;

	memset(&TextAttr, 0, sizeof(struct TextAttr));

	TextAttr.ta_Name = name;

	if (BaseName(name) == name) TextAttr.ta_YSize = defheight;
	else {
		TextAttr.ta_YSize = Atol(BaseName(name));
		*(BaseName(name)-1) = 0;
		if (strcmp(".font", name+strlen(name)-5))
			if (c = ArpAlloc(strlen(name)+6))
				TextAttr.ta_Name = strcat(strcpy(c, name), ".font");
	}

	Font = OpenDiskFont(&TextAttr);
	if (c) FreeTrackedItem((void *)c);
	return(Font);
}

/*
*
* CloseFontSafely()
*
* takes a TextFont ** (a pointer to the font to close) and a FontTable.
* Deallocates the font IF different from FontTable[MAXFONTS].
*
*/

void CloseFontSafely(struct TextFont **Font, struct TextFont **FontTable) {

	if (FontTable[MAXFONTS] != *Font) {
		CloseFont(*Font);
		*Font = FontTable[MAXFONTS];
	}
}

/*
*
* ChangeDefaultFont()
*
* changes the default font with the given (by name) one in D->Arg. Returns
* the new Font address, or NULL if it fails.
*
*/

struct TextFont *ChangeDefaultFont(DisplayStatus *D) {

	struct TextFont *Font;
	int i;

	if (Font = OpenFontNamed(D->Arg, D->FontTable[MAXFONTS]->tf_YSize)) {

		CloseFont(D->FontTable[MAXFONTS]);
		for(i=0; i<=MAXFONTS; i++)
			if (D->FontTable[i] == D->FontTable[MAXFONTS]) D->FontTable[i] = Font;
		SetFont(D->Window->RPort, Font);
	}

	return(Font);
}

/*
*
* CloseAllFonts()
*
* closes all fonts in a FontTable.
*
*/

void CloseAllFonts(struct TextFont **FontTable) {

	int i;
	for(i=0; i<MAXFONTS; i++) CloseFontSafely(&FontTable[i], FontTable);
}


/*
*
* ResetFonts()
*
* closes all open fonts in the DisplayStatus and resets the table to the system font.
*
*/

void ResetFonts(DisplayStatus *D) {

	CloseAllFonts(D->FontTable);
	CloseFont(D->FontTable[MAXFONTS]);
	OpenSystemFont(D->FontTable);
}

/*
*
* ChangeFontAndColor()
*
* changes fonts and colors.
*
*/

void ChangeFontAndColor(DisplayStatus *D, char Font, char Color) {

	if (D->FontTable[FONTNUMBER(Font)] != D->Window->RPort->Font) SetFont(D->Window->RPort, D->FontTable[FONTNUMBER(Font)]);
	if (STYLE(D->Window->RPort->AlgoStyle) != STYLE(Font)) SetSoftStyle(D->Window->RPort, STYLE(Font), 7);

	if (D->Window->RPort->FgPen != FOREGROUND(Color)) SetAPen(D->Window->RPort, FOREGROUND(Color));
	if (D->Window->RPort->BgPen != BACKGROUND(Color)) SetBPen(D->Window->RPort, BACKGROUND(Color));
}

/*
*
* WriteLine()
*
* takes a pointer to a DisplayStatus, to a LineDesc of Leggi's internal format and
* a vertical position. It prints the line at the vertical position
* given, offsetting from borders and adding SKIPTOP to the y coordinate.
*
*/

__asm int nextstr(register __a0 char *);
__asm long skipneg(register __a0 char *);

void WriteLine(DisplayStatus *D, LDesc *LineDesc, int pos) {

	register char *l;
	register int StartX = D->Window->BorderLeft+1-D->LMargin, n;

	if (!(l = LineDesc->Line) || (LineDesc>=D->Buffer+D->LineLen)) return;
	Move(D->Window->RPort, StartX, D->Window->BorderTop+SKIPTOP+pos+LineDesc->Baseline);

	ChangeFontAndColor(D, LineDesc->Font, LineDesc->Color);

	while(l-LineDesc->Line<LineDesc->Length) {

		if (!*l) {
			ChangeFontAndColor(D, *(l+1), *(l+2));
			l = (char *)skipneg(l+3);
		}
		else if (*l == TABCHAR) {
			Move(D->Window->RPort, ((D->Window->RPort->cp_x-StartX)/(D->TabSize*D->Window->RPort->Font->tf_XSize)+1)*(D->TabSize*D->Window->RPort->Font->tf_XSize)+StartX, D->Window->RPort->cp_y);
			l++;
		}
		else {
			n = nextstr(l);
			Text(D->Window->RPort, l, n);
			l += n;
		}
   }
}

/*
*
* FindBottomLine()
*
* takes a DisplayStatus pointer and a LineDesc pointer: returns the bottom LineDesc.
*
*/

LDesc *FindBottomLine(DisplayStatus *D, LDesc *LineDesc) {
	return(GetPixelAddress(LineDesc, D->WindowPixLen));
}


/*
*
* DisplayLines()
*
* takes a pointer to a DisplayStatus and fills it with text.
*
*/

void DisplayLines(DisplayStatus *D) {

	int n=0;
	register LDesc *L=D->TLine;

	do {
		WriteLine(D, L, n);
		n += L->Height;
	} while(D->BLine > ++L);

}

/*
*
* RefreshWindow()
*
* is the same as DisplayLines() but it refreshes it correctly through
* Begin/EndRefresh. If the WipeWindow is set to 1, doesn't do the refresh...
*
*/

void RefreshWindow(DisplayStatus *D) {

	BeginRefresh(D->Window);
	if (!D->DoWipe) {
		DisplayLines(D);
		memset(D->RefreshState, 1, D->WindowLines);
	}
	EndRefresh(D->Window, TRUE);
}


/*
*
* ScrollUpLines()
*
* takes a pointer to DisplayStatus and scrolls up its window basing itself
* on the TLine-OldTLine difference.
*
*/

void ScrollUpLines(DisplayStatus *D) {

	register int i, ScrollBottom;

	for(i=D->WindowLines; (i>0) && (!D->RefreshState[i-1]); i--);
	if (i==0) return;
	ScrollBottom = (i==D->WindowLines ?  D->Window->Height-D->Window->BorderBottom-1 : D->Window->BorderTop+SKIPTOP+GetPixelValue(D->OldTLine, D->OldTLine+i+1)-1);

	if ((D->OldTLine+1 == D->TLine) && (!(D->PropInfo.Flags & KNOBHIT)) && (D->Options & SMOOTHSCROLLING)) {
		for(i=0; i<D->OldTLine->Height; i++) ScrollRaster(D->Window->RPort, 0, 1, 0, D->Window->BorderTop+SKIPTOP, D->Window->Width, ScrollBottom);
	}
	else ScrollRaster(D->Window->RPort, 0, GetPixelValue(D->OldTLine, D->TLine), 0, D->Window->BorderTop+SKIPTOP, D->Window->Width, ScrollBottom);

	movmem(D->RefreshState+(D->TLine-D->OldTLine), D->RefreshState, D->WindowLines);

}


/*
*
* ScrollDownLines()
*
* same as ScrollUpLines(), but scrolls down.
*
*/

void ScrollDownLines(DisplayStatus *D) {

	register int i, ScrollTop;

	for(i=0; (i<D->WindowLines) && (!D->RefreshState[i]); i--);
	if (i==D->WindowLines) return;
	ScrollTop = D->Window->BorderTop+SKIPTOP+GetPixelValue(D->OldTLine, D->OldTLine+i);

	if ((D->OldTLine == D->TLine+1) && (!(D->PropInfo.Flags & KNOBHIT)) && (D->Options & SMOOTHSCROLLING)) {
		for(i=0; i<D->TLine->Height; i++) ScrollRaster(D->Window->RPort, 0, -1, 0, ScrollTop, D->Window->Width, D->Window->Height-D->Window->BorderBottom);
	}
	else ScrollRaster(D->Window->RPort, 0, -GetPixelValue(D->TLine, D->OldTLine), 0, ScrollTop, D->Window->Width, D->Window->Height-D->Window->BorderBottom);

	movmem(D->RefreshState, D->RefreshState+(D->OldTLine-D->TLine), D->WindowLines-(D->OldTLine-D->TLine));
	memset(D->RefreshState, 0, D->OldTLine-D->TLine);

	SetAPen(D->Window->RPort, 0);
	if (D->BLine>D->TLine) RectFill(D->Window->RPort, D->Window->BorderLeft, D->Window->BorderTop+SKIPTOP+GetPixelValue(D->TLine, FindBottomLine(D, D->TLine)), D->Window->Width, D->Window->Height);
}

/*
*
* WipeWindow()
*
* takes a DisplayStatus pointer and wipes it, IE cancels the contents and sets
* up a refresh.
*
*/

void WipeWindow(DisplayStatus *D) {
	D->DoWipe = 1;
	if (!D->DoRefresh) D->DoRefresh = 1;
}

/*
*
* ResetWindowTitle()
*
* really a stupid function, but it's soooo used...
*
*/

void ResetWindowTitle(struct Window *Window) {

	SetWindowTitles(Window, Window->Title, (void *)(-1));
}

/*
*
* StripRefreshMessages()
*
* takes a Window pointer and strips all messages directed to that window
* of class REFRESHWINDOW. If some are present, does a Begin/EndRefresh()
* to keep things in order.
*
*/

void StripRefreshMessages(struct Window *Window) {

 	struct IntuiMessage *Message, *Succ;
	int f=0;

	Message = (void *)Window->UserPort->mp_MsgList.lh_Head;

	Forbid();

	while(Succ = (void *)Message->ExecMessage.mn_Node.ln_Succ) {

		if (Message->IDCMPWindow == Window && Message->Class == REFRESHWINDOW) {
			f = 1;
			Remove((void *)Message);
			ReplyMsg((void *)Message);
		}

		Message = Succ;
	}

	Permit();

	if (f) {
		BeginRefresh(Window);
		EndRefresh(Window, TRUE);
	}
}
