#include "SD.h"

#define NO_DMA_PENDING 0
#define READ_PENDING 1
#define WRITE_PENDING 2

static unsigned char TimerName[] = TIMERNAME,
							RecalibrateErrorMsg[] = "Error while recalibrating unit x.",
							ClickPending,	/* Is there any click pending? */
							DMAPending,		/* Is there any DMA operation pending? */
							DMAIndex,		/* What DMA check are we currently using? */
							HeadMoving,		/* Is there any head move pending? */
							*DMAWriteBuffer,	/* What's the current DMA write buffer? */
							*DMAReadBuffer;	/* What's the current DMA read buffer? */


static unsigned short 	CurrentCyl[4],				/* where we are */
								BackedupTrack;				/* For retries */
struct TDU_PublicUnit 	*TDU_PublicUnit[4];		/* The public units. They tell us which
																	drives have been opened by trackdisk. */

static struct Interrupt SoftInt = { { NULL, NULL, NT_INTERRUPT, 32, NULL }, (APTR)(&SoftIntData), SoftIntCode };
static struct Interrupt SoftInt2 = { { NULL, NULL, NT_INTERRUPT, 32, NULL }, (APTR)(&SoftIntData), SoftIntCode2 };
static struct MsgPort SoftIntPort = { { NULL, NULL, NT_MSGPORT, 0, NULL }, PA_SOFTINT, 0, (struct Task *)(&SoftInt2) };

static struct Interrupt SyncInt = { { NULL, NULL, NT_INTERRUPT, 32, NULL }, (APTR)(&SyncIntData), SyncIntCode };
static struct Interrupt SyncInt2 = { { NULL, NULL, NT_INTERRUPT, 32, NULL }, (APTR)(&SyncIntData), SyncIntCode2 };
static struct MsgPort SyncIntPort = { { NULL, NULL, NT_MSGPORT, 0, NULL }, PA_SOFTINT, 0, (struct Task *)(&SyncInt2) };

static struct timerequest
	SoftInt_tr = {	{ { { NULL, NULL, NT_MESSAGE, 127, NULL }, &SoftIntPort, sizeof(struct timerequest) } } },	/* For the SoftInt */
	SyncInt_tr = { { { { NULL, NULL, NT_MESSAGE, 127, NULL }, &SyncIntPort, sizeof(struct timerequest) } } },	/* For the SyncInt */
	tr = { { { { NULL, NULL, NT_MESSAGE, 126, NULL }, NULL, sizeof(struct timerequest) } } }, 			/* General purpose */
	HM_tr = { { { { NULL, NULL, NT_MESSAGE, 127, NULL }, NULL, sizeof(struct timerequest) } } }, 		/* For head move timing */
	Click_tr = { { { { NULL, NULL, NT_MESSAGE, 0, NULL }, NULL, sizeof(struct timerequest) } } },		/* For clicking drives */
	CT_tr = { { { { NULL, NULL, NT_MESSAGE, 126, NULL }, NULL, sizeof(struct timerequest) } } };			/* Copy task, general purpose */

static unsigned char tr_open,
							SoftInt_tr_open,
							SyncInt_tr_open,
							HM_tr_open,
							Click_tr_open,
							CT_tr_open;				/* The results of OpenDevice() */

struct SyncIntData volatile SyncIntData = { 1, 1, 0, &SyncIntPort, &SyncInt_tr, &SyncInt };
struct SoftIntData volatile SoftIntData = { { NULL, NULL }, &SoftIntPort, &SoftInt_tr, NULL, 0, 0, 0, &DiskBlockServerData };

/* We open the timer.device for the copy task and set up all our timerequests.

	HM_tr: used for head move timing; it's set by MoveHeads() and waited by WaitHeads().

	SoftInt_tr: used by the SoftInt in order to perform the side select delay; generates
					the SoftInt on completion.

	SyncInt_tr: used by the SyncInt to skip 100us before a new sync is counted.

	CT_tr: used by the copy task for generic sync waits, such as the side select
				delay in DMA.a

	These timerequests can *only* be used from the copy task!
*/

unsigned char CopyTaskOpenTimer(void) {

	if ((CT_tr.tr_node.io_Message.mn_ReplyPort = HM_tr.tr_node.io_Message.mn_ReplyPort = CreateMsgPort()) == NULL) return(FALSE);
	NewList(&(SoftIntPort.mp_MsgList));
	NewList(&(SyncIntPort.mp_MsgList));

	HM_tr_open = !OpenDevice(TimerName, UNIT_MICROHZ, &HM_tr, 0);
	SoftInt_tr_open = !OpenDevice(TimerName, UNIT_MICROHZ, &SoftInt_tr, 0);
	SyncInt_tr_open = !OpenDevice(TimerName, UNIT_MICROHZ, &SyncInt_tr, 0);
	CT_tr_open = !OpenDevice(TimerName, UNIT_MICROHZ, &CT_tr, 0);
	SyncInt_tr.tr_node.io_Command = SoftInt_tr.tr_node.io_Command = TR_ADDREQUEST;
	return((unsigned char)(HM_tr_open && SoftInt_tr_open && SyncInt_tr_open && CT_tr_open));
}



/* Here we close everything for the copy task. */

void CopyTaskCloseTimer(void) {

	if (HM_tr_open) CloseDevice(&HM_tr);
	if (SoftInt_tr_open) CloseDevice(&SoftInt_tr);
	if (SyncInt_tr_open) CloseDevice(&SyncInt_tr);
	if (CT_tr_open) CloseDevice(&CT_tr);
	DeleteMsgPort(HM_tr.tr_node.io_Message.mn_ReplyPort);
}


/* We open the timer.device for the main task and set up all our timerequests.

	tr: used by the main task for generic sync waits, such as step delays in ResetHeads().

	Click_tr: used for periodically checking the drives.

	These timerequests can *only* be used from the main task! Moreover, ClickPort *must*
	point to a regularly opened MsgPort.
*/

unsigned char OpenTimer(void) {

	tr.tr_node.io_Message.mn_ReplyPort = Click_tr.tr_node.io_Message.mn_ReplyPort = ClickPort;

	tr_open = !OpenDevice(TimerName, UNIT_MICROHZ, &tr, 0);
	Click_tr_open = !OpenDevice(TimerName, UNIT_VBLANK, &Click_tr, 0);

	TimerBase = (void *)tr.tr_node.io_Device;
	return((unsigned char)(tr_open && Click_tr_open));
}

/* Here we close everything for the main task. */

void CloseTimer(void) {
	if (tr_open) CloseDevice(&tr);
	if (Click_tr_open) CloseDevice(&Click_tr);
}

/* Wait for the specified amount of s. There are two versions,
	one for the copy task, and it uses CT_tr; the other one for the main
	task, and it uses tr. */


static void GlobTimerWait(ULONG microseconds, struct timerequest *timereq) {

	timereq->tr_node.io_Command = TR_ADDREQUEST;
	timereq->tr_time.tv_secs = 0;
	timereq->tr_time.tv_micro = microseconds;
	DoIO(timereq);
}

void TimerWait(ULONG microseconds) {
	GlobTimerWait(microseconds, &tr);
}

void __asm __stdargs CopyTaskTimerWait(register __d0 ULONG microseconds) {
	GlobTimerWait(microseconds, &CT_tr);
}

/* Generic routine to SendIO a timerequest. Works from any task. */

static void TimerPost(struct timerequest *t, ULONG microseconds) {
	t->tr_node.io_Command = TR_ADDREQUEST;
	t->tr_time.tv_secs = 0;
	t->tr_time.tv_micro = microseconds;
	SendIO(t);
}

/* Posting/Removing the timerequest for disk clicks. Only from the main task. */

void PostClick(void) {
	if (!ClickPending) {
		TimerPost(&Click_tr, 100000UL);
		ClickPending = 1;
	}
}

void RemoveClick(void) {
	if (ClickPending) {
		AbortIO(&Click_tr);
		WaitIO(&Click_tr);
		ClickPending = 0;
	}
}

/* Is there an expired Click_tr? */

unsigned char CheckClick(void) {
	return((unsigned char)(ClickPending && CheckIO(&Click_tr)));
}


/* We move back to the previous track and store the current one.
	This routine is used by the retry system. Call this only from the copy task. */

void MoveAndBackupTrack(unsigned char Unit, unsigned short Track) {
	ASSERT(Track<160)
	BackedupTrack = CurrentCyl[Unit]*2;
	MoveHead(Unit, Track);
}

void RestoreTrack(unsigned char Unit) {
	ASSERT(BackedupTrack<160)
	MoveHead(Unit, BackedupTrack);
}

/* This routine trashes the first bytes of a buffer (otherwise
	a completely empty read could pass unnoticed) */

static void TrashBuffer(unsigned char *Buffer) {
	*((ULONG *)(Buffer-8)) = *((ULONG *)(Buffer-4)) = *((ULONG *)(Buffer)) = *((ULONG *)(Buffer+4)) = 0xFFFFFFFF;
}

void MoveHead(unsigned char Unit, unsigned short Track) {

	register unsigned char Mask = MASK(Unit);

	if (Unit>3 || Track>159 || (Track /= 2) == CurrentCyl[Unit]) return;

	WaitDMA();
	if (HeadMoving & Mask) WaitHeads();

	if (HeadMoving) {
		if (CheckIO(&HM_tr)) HeadMoving = 0;
		else {
			AbortIO(&HM_tr);
			CopyTaskTimerWait(StepDelay);
		}
		WaitIO(&HM_tr);
	}

	if (Track == CurrentCyl[Unit]+1) StepHead(Unit, 0);
	else if (Track == CurrentCyl[Unit]-1) StepHead(Unit, CIAF_DSKDIREC);

	TimerPost(&HM_tr, StepDelay+SettleDelay);
	HeadMoving |= Mask;
	CurrentCyl[Unit] = Track;
}

/* Here we move a mask of heads. ONLY one track more or less
	the current one, and ONLY from the copy task! */

void MoveHeads(unsigned char Mask, unsigned short Track) {
	unsigned char i;
	for(i=0; i<4; i++) if (INMASK(i,Mask)) MoveHead(i, Track);
}

/* These two functions provide a temporary priority rising, with following lowering. They
	can be called *only* *once* (no nesting) from the main task. */

static signed char OldPri;

static void LoPri(void) {
	SetTaskPri(Me, OldPri);
}

static void HiPri(void) {
	if ((OldPri = SetTaskPri(Me, 15))>15) LoPri();
}

/* Here we move a head mask to a given track. Only from the main task. Uses Hi/LoPri. */

void ResetHeads(unsigned char Mask, unsigned short Track, unsigned char Flags) {

	register unsigned char i;

	Track /= 2;

	HiPri();
	for(i=0; i<4; i++)
		if (INMASK(i,Mask) && CurrentCyl[i] != Track) {
			while(CurrentCyl[i] != Track) {
				if (CurrentCyl[i] < Track) {
					StepHead(i, Flags);
					CurrentCyl[i]++;
				}
				else {
					StepHead(i, CIAF_DSKDIREC | Flags);
					CurrentCyl[i]--;
				}
				TimerWait(StepDelay);
			}
		}
	LoPri();

	TimerWait(SettleDelay);
}

/* Here we move a single head to a given track. We first check for a disk
	being inserted and turn on the motor accordingly. The motor is *NOT* turned off.
	We return the CIA flags. */

unsigned char ResetHead(unsigned char Unit, unsigned short Track) {
	unsigned char Flags;

	if (GetReallyCIAAPRA(Unit) & CIAF_DSKCHANGE) {
		MotorOn(MASK(Unit));
		TimerWait(500000);
		Flags = 0;
	}
	else Flags = CIAF_DSKMOTOR;
	ResetHeads(MASK(Unit), Track, Flags);
	return(GetCIAAPRA(Unit, Flags, TRUE));
}


/* Here we recalibrate the head of a unit. We move the head to track 0, then
	we step back and forth of 1 track. Only from the main task. Uses Hi/LoPri(). */

void RecalibrateHead(unsigned char Unit) {

	unsigned char CIA;

	CIA = ResetHead(Unit, 0);
/*	if (!(CIA & CIAF_DSKCHANGE)) {
	StepHead(Unit, CIAF_DSKMOTOR);
	TimerWait(StepDelay+SettleDelay);
	StepHead(Unit, CIAF_DSKDIREC | CIAF_DSKMOTOR);
	TimerWait(StepDelay+SettleDelay);*/
	if ((GetCIAAPRA(Unit, (CIA & CIAF_DSKCHANGE) ? 0 : CIAF_DSKMOTOR, FALSE) & CIAF_DSKTRACK0) &&
		(GetCIAAPRA(Unit, 0, TRUE) & CIAF_DSKTRACK0)) {
		CIA = 0;
		do {
			StepHead(Unit, CIAF_DSKDIREC | CIAF_DSKMOTOR);
			TimerWait(CalibrateDelay);
		} while((GetCIAAPRA(Unit, 0, TRUE) & CIAF_DSKTRACK0) && CIA++<100);

		ASSERT(CIA<80);
		RecalibrateErrorMsg[31] = Unit+48;
		if (RecalibrateCheck) Acknowledge(RecalibrateErrorMsg);
	}
	MotorOff(MASK(Unit));
}

/* Here we compute the position of the head of a unit by recalibrating it. */

void FindTrack(unsigned char Unit) {

	unsigned char Flags;
	unsigned short Cyl = 0;

	TimerWait(CalibrateDelay+SettleDelay);
	if (GetCIAAPRA(Unit, CIAF_DSKMOTOR, FALSE) & CIAF_DSKTRACK0) {
		StepHead(Unit, CIAF_DSKMOTOR | CIAF_DSKDIREC);
		TimerWait(CalibrateDelay+SettleDelay);
		Cyl = 1;
		if ((Flags = GetCIAAPRA(Unit, CIAF_DSKMOTOR, TRUE)) & CIAF_DSKTRACK0) {
			if (Flags & CIAF_DSKCHANGE) {
				MotorOn(MASK(Unit));
				TimerWait(500000);
				Flags = 0;
			}
			else Flags = CIAF_DSKMOTOR;
			HiPri();
			while((GetCIAAPRA(Unit, Flags, FALSE) & CIAF_DSKTRACK0) && Cyl<100) {
				StepHead(Unit, CIAF_DSKDIREC | Flags);
				TimerWait(CalibrateDelay);
				Cyl++;
			}
			LoPri();
			MotorOff(MASK(Unit));
		}
		TimerWait(SettleDelay);
	}

	ASSERT(Cyl<80);
	CurrentCyl[Unit] = 0;
}


/* Click the heads of the selected drive if necessary to get a correct CIA reading. */

void ClickHeads(unsigned char Mask, unsigned long ticks) {

	unsigned char Unit, f = 0;

	for(Unit=0; Unit<4; Unit++)
		if (INMASK(Unit,Mask) && (!(ticks%15) || ((TDU_PublicUnit[Unit] && (TDU_PublicUnit[Unit]->tdu_PubFlags & TDPF_NOCLICK)) && !(ticks%5)))) {
			GetReallyCIAAPRA(Unit);
			f = 1;
		}
	if (f) TimerWait(SettleDelay);
}

/* Here we get the "real" value of the CIA register, i.e.,
	if DSKCHANGE is 0 we step the heads and check again.
	We wait for a step delay, and NOT for the settle delay.
	The motor is kept OFF. Only from the main task. */

unsigned char GetReallyCIAAPRA(unsigned char Unit) {

	unsigned char CIA;
	if ((CIA = GetCIAAPRA(Unit, CIAF_DSKMOTOR, TRUE)) & CIAF_DSKCHANGE) return(CIA);
	if (TDU_PublicUnit[Unit] && (TDU_PublicUnit[Unit]->tdu_PubFlags & TDPF_NOCLICK)) {
		StepHead(Unit, CIAF_DSKMOTOR | CIAF_DSKDIREC);
		if (CurrentCyl[Unit]) CurrentCyl[Unit]--;
	}
	else {
		StepHead(Unit, CIAF_DSKMOTOR | (CurrentCyl[Unit] ? CIAF_DSKDIREC : 0));
		if (CurrentCyl[Unit]) CurrentCyl[Unit]--; else CurrentCyl[Unit] = 1;
	}
	TimerWait(StepDelay);
	return(GetCIAAPRA(Unit, CIAF_DSKMOTOR, TRUE));
}

/* Here we start a DMA write on the given destination. We specify the track
	for precomp reasons, and some stuff to start the optimized verify. Only from the copy task. */

void StartWrite(unsigned char DestMask, unsigned short Track, unsigned char *Buffer, unsigned short Size, unsigned char VerifyDrive, unsigned char *VBuffer, unsigned char GetSync) {

	unsigned char PreComp = 0;

	ASSERT(Track<160)
	ASSERT(Buffer)

	if (Track>159) return;

	WaitDMA();

	if (Track>=TDU_PublicUnit[0]->tdu_Comp01Track) PreComp = 1;
	if (Track>=TDU_PublicUnit[0]->tdu_Comp10Track) PreComp = 2;
	if (Track>=TDU_PublicUnit[0]->tdu_Comp11Track) PreComp = 3;

	DiskBlockServerData.DBSD_SoftInt = &SoftInt;
	SoftInt_tr.tr_time.tv_secs = 0;
	SoftInt_tr.tr_time.tv_micro = 2000;

	if (SoftIntData.SID_VerifyBuffer = VBuffer) {
		DMAPending = READ_PENDING;
		TrashBuffer(VBuffer);
		SoftIntData.SID_VerifyLength = VerifyLength+GetSync*GapSize;
		SyncIntData.SD_SecCounter = -((short int)SPT+1);
		SyncIntData.SD_GetSync = GetSync;
		SoftIntData.SID_CIABPRB = ((~(MASK(VerifyDrive)) & 15)<<CIAB_DSKSEL0) | (1-(Track%2))<<CIAB_DSKSIDE | CIAF_DSKSTEP;
		SoftIntData.SID_ADKCON = MASK(ADKB_SETCLR) | ADKF_WORDSYNC | ADKF_FAST | ADKF_MFMPREC | ((ULONG)(ADKF_PRECOMP0 | ADKF_PRECOMP1 | ADKF_MSBSYNC) << 16);
		DMAReadBuffer = VBuffer;
	}
	else {
		DMAPending = WRITE_PENDING;
		DMAReadBuffer = NULL;
	}
	DMAWriteBuffer = Buffer+Size-WriteLength;
	DMAIndex = Track%8;

	if (HeadMoving & DestMask) WaitHeads();
	DMAWrite(Buffer, Size,
		((~DestMask & 15)<<CIAB_DSKSEL0) | (1-(Track%2))<<CIAB_DSKSIDE | CIAF_DSKSTEP,
		(MASK(ADKB_SETCLR) | PreComp<<ADKB_PRECOMP0 |  ADKF_FAST | ADKF_MFMPREC)
		| ((ULONG)((~PreComp & 3)<<ADKB_PRECOMP0 | ADKF_WORDSYNC | ADKF_MSBSYNC) << 16) );
}

/* Here we start a DMA read. Only from the copy task. */

void StartRead(unsigned char Source, unsigned short Track, unsigned char *Buffer, unsigned char GetSync) {

	ASSERT(Track<160)
	ASSERT(Buffer)

	if (Track>159) return;

	WaitDMA();
	DiskBlockServerData.DBSD_SoftInt = NULL;
	TrashBuffer(Buffer);
	DMAWriteBuffer = NULL;
	DMAReadBuffer = Buffer;
	SyncIntData.SD_SecCounter = -((short int)SPT+1);
	SyncIntData.SD_GetSync = GetSync;
	if (HeadMoving & MASK(Source)) WaitHeads();
	DMARead(Buffer, ReadLength+GapSize,
		((~(MASK(Source)) & 15)<<CIAB_DSKSEL0) | (1-(Track%2))<<CIAB_DSKSIDE | CIAF_DSKSTEP,
		(ULONG)(MASK(ADKB_SETCLR) | ADKF_WORDSYNC |  ADKF_FAST | ADKF_MFMPREC)
		| ((ULONG)(ADKF_PRECOMP0 | ADKF_PRECOMP1 | ADKF_MSBSYNC) << 16) );
	DMAPending = READ_PENDING;
}



/* Here we wait for the completion of a DMA operation. We set up a timer
	so that if WORDSYNC never starts we are alerted. If the data in the buffer
	does not correspond to the checksum we alert of a munged buffer. Only from the copy task. */

void WaitDMA(void) {

	ULONG WaitMask, Signals;

	if (DMAPending != NO_DMA_PENDING) {
		WaitMask = SoftIntData.SID_ISD.ISD_Mask;
		if (DMAPending == READ_PENDING) {
			TimerPost(&CT_tr, 999990);
			WaitMask |= (MASK(CT_tr.tr_node.io_Message.mn_ReplyPort->mp_SigBit));
		}
		if (DMAWriteBuffer && CheckTrack(DMAWriteBuffer, SPT*BLOCKSIZE) != DMACheck[DMAIndex]) MungedBuffer = 1;

		ASSERT(!MungedBuffer)
		ASSERT(!(SetSignal(0,0) & SoftIntData.SID_ISD.ISD_Mask))

		do Signals = Wait(WaitMask);
		while(!(Signals & SoftIntData.SID_ISD.ISD_Mask) && !(DMAPending == READ_PENDING && CheckIO(&CT_tr)));
		if (DMAPending == READ_PENDING) {
			if (Signals & SoftIntData.SID_ISD.ISD_Mask) AbortIO(&CT_tr);
			else {
				StopDMA();
				if (DMAReadBuffer) {
					BltClear(DMAReadBuffer, ReadLength+GapSize, 0);
					TrashBuffer(DMAReadBuffer);
				}
			}
			WaitIO(&CT_tr);
		}
		ClearInterrupts();
		DMAPending = NO_DMA_PENDING;
	}

	ASSERT(SyncIntData.SD_EnableSecCounter)
	while(!SyncIntData.SD_EnableSecCounter) CopyTaskTimerWait(100);
}

/* We wait until the heads are moved. Only from the copy task. */

void WaitHeads(void) {

	if (HeadMoving) {
		WaitIO(&HM_tr);
		HeadMoving = 0;
	}
}

/* Two-part disk preparation routine called from SD.c. Only from the main task. */

void SetUpDisksA(void) {
	TimerPost(&tr, 500000);
}

void SetUpDisksB(unsigned char UnitMask, unsigned short Track) {

	ASSERT(!CheckIO(&tr))

	WaitIO(&tr);
	ResetHeads(UnitMask, Track, 0);
}

void SetPublicUnit(unsigned char Unit) {

	ASSERT(CurrentCyl[Unit]<80)

	TDU_PublicUnit[Unit]->tdu_CurrTrk = CurrentCyl[Unit]*2;
}
