#include "SD.h"
#include <rexx/errors.h>

#define MAXPAR 10
#define RDABUFFERSIZE 1024
#define TEMPBUFFERSIZE 1024

struct RxsLib *RexxSysBase;

static struct RDArgs *RDA, *RA;
static char *RDABuffer, *Temp;
static unsigned long Par[MAXPAR];
static BPTR ARexxConsole;
static struct RexxMsg *StartupMsg;

static unsigned char ARexxConsoleName[] = "CON:0/2/512/128/SuperDuper ARexx Console/AUTO/CLOSE",
							NoHostMsg[] = "ARexx server not active.",
							ARexxErrorMsg[] = "ARexx error: ",
							NotValidMsg[] = "Unknown",
							CantOpenStartupFileMsg[] = "Can't open startup file.";

static CommandData Comm[] = {
	{	STOP_COMMAND, 		NO_TEMPLATE, sizeof(STOP_COMMAND)-1,	STOP,	COMMF_NOARGS	},
	{	COPY_COMMAND, 		NO_TEMPLATE, sizeof(COPY_COMMAND)-1,	COPY,	COMMF_NOARGS	},
	{	READ_COMMAND, 		NO_TEMPLATE, sizeof(READ_COMMAND)-1,	READ,	COMMF_NOARGS	},
	{	WRITE_COMMAND, 	NO_TEMPLATE, sizeof(WRITE_COMMAND)-1,	WRITE,	COMMF_NOARGS	},
	{	CHECK_COMMAND, 	NO_TEMPLATE, sizeof(CHECK_COMMAND)-1,	CHECK,	COMMF_NOARGS	},
	{	FORMAT_COMMAND, 	NO_TEMPLATE, sizeof(FORMAT_COMMAND)-1,	FORMAT,	COMMF_NOARGS	},
	{	AREXX_COMMAND, 	NO_TEMPLATE, sizeof(AREXX_COMMAND)-1,	AREXX,	COMMF_NOARGS	},
	{	NOWB_COMMAND,		ONOFF_TEMPLATE, sizeof(NOWB_COMMAND)-1,	NOWB,	0	},
	{	ICONIFY_COMMAND, 	ONOFF_TEMPLATE, sizeof(ICONIFY_COMMAND)-1,	ICONIFY,	0	},
	{	SAVECONF_COMMAND,	NO_TEMPLATE, sizeof(SAVECONF_COMMAND)-1,	SAVECONF,	COMMF_NOARGS	},
	{	VERIFY_COMMAND,	ONOFF_TEMPLATE, sizeof(VERIFY_COMMAND)-1,	VERIFY,	0	},
	{	DATE_COMMAND,		ONOFF_TEMPLATE, sizeof(DATE_COMMAND)-1,	DATE,	0	},
	{	COMP_COMMAND,		ONOFF_TEMPLATE, sizeof(COMP_COMMAND)-1,	COMPRESSION,	0	},
	{	PRINTERRORS_COMMAND,	ONOFF_TEMPLATE,	sizeof(PRINTERRORS_COMMAND)-1,	PRINTERRORS,	0	},
	{	INCNAME_COMMAND,	ONOFF_TEMPLATE, sizeof(INCNAME_COMMAND)-1,	INCNAME,	0	},
	{	FFS_COMMAND,		ONOFF_TEMPLATE, sizeof(FFS_COMMAND)-1,	FFS,	0	},
	{	INTL_COMMAND,		ONOFF_TEMPLATE, sizeof(INTL_COMMAND)-1,	INTL,	0	},
	{	DIRCACHE_COMMAND,	ONOFF_TEMPLATE, sizeof(DIRCACHE_COMMAND)-1,	DIRCACHE,	0	},
	{	TALK_COMMAND,		ONOFF_TEMPLATE, sizeof(TALK_COMMAND)-1,	TALK,	0	},
	{	AUTO_COMMAND,		ONOFF_TEMPLATE, sizeof(AUTO_COMMAND)-1,	AUTO,	0	},
	{	LABEL_COMMAND,		STRING_TEMPLATE, sizeof(LABEL_COMMAND)-1,	LABEL,	0	},
	{	FILENAME_COMMAND, STRING_TEMPLATE,	sizeof(FILENAME_COMMAND)-1,	FILENAME,	0	},
	{	XPKLIB_COMMAND,	STRING_TEMPLATE,	sizeof(XPKLIB_COMMAND)-1,	XPKLIB,	0	},
	{	VDNAME_COMMAND,	STRING_TEMPLATE,	sizeof(VDNAME_COMMAND)-1,	VDNAME,	0	},
	{	VDNUMBER_COMMAND,	NUMBER_TEMPLATE,	sizeof(VDNUMBER_COMMAND)-1,	VDNUMBER,	0	},
	{	RETRY_COMMAND,		NUMBER_TEMPLATE,	sizeof(RETRY_COMMAND)-1,	RETRY,	0	},
	{	SCYL_COMMAND,		NUMBER_TEMPLATE,	sizeof(SCYL_COMMAND)-1,	SCYL,	0	},
	{	ECYL_COMMAND,		NUMBER_TEMPLATE,	sizeof(ECYL_COMMAND)-1,	ECYL,	0	},
	{	COPYMODE_COMMAND,	COPYMODE_TEMPLATE, sizeof(COPYMODE_COMMAND)-1,	COPYMODE,	0	},
	{	QUIT_COMMAND,		NO_TEMPLATE,		sizeof(QUIT_COMMAND)-1,	QUIT,	COMMF_NOARGS	},
	{	REQUESTERS_COMMAND,	ONOFF_TEMPLATE,	sizeof(REQUESTERS_COMMAND)-1,	REQUESTERS,	0	},
	{	RECALIBRATECHECK_COMMAND,	ONOFF_TEMPLATE,	sizeof(RECALIBRATECHECK_COMMAND)-1,	RECALIBRATECHECK,	0	},
	{	SMARTREFRESH_COMMAND,	ONOFF_TEMPLATE,	sizeof(SMARTREFRESH_COMMAND)-1,	SMARTREFRESH,	0	},
	{	DESTINATION_COMMAND,	DESTINATION_TEMPLATE, sizeof(DESTINATION_COMMAND)-1,	DESTINATION,	0	},
	{	SOURCE_COMMAND,	SOURCE_TEMPLATE, sizeof(SOURCE_COMMAND)-1,	SOURCE,	0	},
	{	HELP_COMMAND, HELP_TEMPLATE, sizeof(HELP_COMMAND)-1,	HELP,	COMMF_NOARGS	},
	{	NOP_COMMAND, NO_TEMPLATE, sizeof(NOP_COMMAND)-1,	NOP,	COMMF_NOARGS },
	{	RX_COMMAND, RX_TEMPLATE, sizeof(RX_COMMAND)-1,	NOP,	0	},
	{	WINDOW_COMMAND,	WINDOW_TEMPLATE,	sizeof(WINDOW_COMMAND)-1,	WINDOW, 0 }
};

static void ReleaseRexxArgs(void) {

	if (RA) FreeArgs(RA);
	RA = NULL;
}

unsigned short InitARexx(void) {

	if (!FindPort(ARexxPortName) &&
		(RexxSysBase = (void *)OpenLibrary(RXSNAME, 0)) &&
		(RDA = AllocDosObject(DOS_RDARGS, NULL)) &&
		(RDABuffer = AllocVec(RDABUFFERSIZE, MEMF_PUBLIC)) &&
		(Temp = AllocVec(TEMPBUFFERSIZE, MEMF_PUBLIC)) &&
		(ARexxPort = CreateMsgPort())) {
		ARexxPort->mp_Node.ln_Name = ARexxPortName;
		AddPort(ARexxPort);
		return(TRUE);
	}
	else {
		CloseARexx();
		return(FALSE);
	}
}

void CloseARexx(void) {

	struct RexxMsg *ARexxMessage;

	if (ARexxPort) {

		ASSERT(FindPort(ARexxPortName));

		Forbid();
		while(ARexxMessage=(struct RexxMsg *)GetMsg(ARexxPort)) {
			ARexxMessage->rm_Result1 = 30;
			ARexxMessage->rm_Result2 = NULL;
			ReplyMsg((struct Message *)ARexxMessage);
		}
		RemPort(ARexxPort);
		Permit();
		DeleteMsgPort(ARexxPort);
	}
	ReleaseRexxArgs();
	if (RDA) FreeDosObject(DOS_RDARGS, RDA);
	FreeVec(RDABuffer);
	FreeVec(Temp);
	CloseLibrary((void *)RexxSysBase);
	ASLFreeFileReq();
}

void SetRexxError(struct RexxMsg *RexxMsg, int RC) {

	if (!RexxMsg) return;
	RexxMsg->rm_Result1 = RC;
	RexxMsg->rm_Result2 = NULL;
}

void SetRexxReturn(struct RexxMsg *RexxMsg, char *Result) {

	if (!RexxMsg) return;
	RexxMsg->rm_Result1 = 0;
	RexxMsg->rm_Result2 = ((RexxMsg->rm_Action & RXFF_RESULT) && Result) ? (LONG)CreateArgstring(Result, strlen(Result)+1) : NULL;
}

void SetRexxReturnN(struct RexxMsg *RexxMsg, int Result) {
	char t[16];

	if (!RexxMsg) return;
	sprintf(t, "%ld", (ULONG)Result);
	RexxMsg->rm_Result1 = 0;
	RexxMsg->rm_Result2 = (RexxMsg->rm_Action & RXFF_RESULT) ? (LONG)CreateArgstring(t, strlen(t)) : NULL;
}

void PrintARexxHelp(struct RexxMsg *RexxMsg, char *Command) {

	unsigned short i,j;

	ASSERT(RexxMsg);

	if (Command) {
		for(i=0; i<sizeof(Comm)/sizeof(CommandData); i++)
			if (Strnicmp(Comm[i].Name, Command, strlen(Command)) == 0) {
				SetRexxReturn(RexxMsg, Comm[i].Template);
				return;
			}
		if (i == sizeof(Comm)/sizeof(CommandData)) SetRexxError(RexxMsg, 5);
	}
	else {
		PutStr("Command Name\t\tTemplate\n");
		for(i=0; i<64; i++) PutStr("-");
		PutStr("\n");
		for(i=0; i<sizeof(Comm)/sizeof(CommandData); i++) {
			PutStr(Comm[i].Name);
			for(j=0; j<3-Comm[i].NameLen/8; j++) PutStr("\t");
			PutStr(Comm[i].Template);
			PutStr("\n");
		}
	}
}


/* This routine launches an ARexx macro selected by the user */


unsigned char StartARexxMacro(unsigned char *MacroName, unsigned char IsStartup) {

	struct RexxMsg *RexxMsg;
	struct MsgPort *HostPort;

	if (MacroName) {
		if (RexxMsg = CreateRexxMsg(ARexxPort, &DefaultPattern[3], ARexxPortName)) {
			if (stdin && stdout) RexxMsg->rm_Action = RXCOMM;
			else if (ARexxConsole || (ARexxConsole = Open(ARexxConsoleName, MODE_NEWFILE))) {
				RexxMsg->rm_Action = RXCOMM | MASK(RXFB_NOIO);
				RexxMsg->rm_Stdin = RexxMsg->rm_Stdout = ARexxConsole;
			}
			else goto Failure;
			RexxMsg->rm_Args[0] = MacroName;
			Forbid();
			if (HostPort = FindPort("REXX")) PutMsg(HostPort, (void *)RexxMsg);
			Permit();
			if (HostPort) {
				if (IsStartup) StartupMsg = RexxMsg;
				return(1);
			}
			if (!IsStartup) Acknowledge(NoHostMsg);
		}
	}
	else return(0);
Failure:
	DeleteArgstring(MacroName);
	if (RexxMsg) DeleteRexxMsg(RexxMsg);
	return(0);
}

void CloseARexxConsole(void) {
	if (ARexxConsole) {
		Close(ARexxConsole);
		ARexxConsole = NULL;
	}
}

/* After an arexx macro... */

int __stdargs ErrorMsg(LONG, struct NexxStr **);

void AfterARexxMacro(struct RexxMsg *RexxMsg) {

	struct NexxStr *p;

	if (RexxMsg->rm_Result1 && (StartupMsg != RexxMsg || RexxMsg->rm_Result2 != ERR10_001)) {
		ObtainSemaphore(&Semaphore[WIN_OPTIONS]);
		strcpy(s, ARexxErrorMsg);
		if (ErrorMsg(RexxMsg->rm_Result2, (void *)&p)) strcat(s, p->ns_Buff);
		else strcat(s, NotValidMsg);
		Acknowledge(s);
		ReleaseSemaphore(&Semaphore[WIN_OPTIONS]);
	}
	DeleteArgstring(RexxMsg->rm_Args[0]);
	DeleteRexxMsg(RexxMsg);
	if (RexxMsg == StartupMsg) StartupMsg = NULL;
}


/*
*
* ParseARexx()
*
* takes an ARexxMessage pointer and the address of a pointer which will
* be set to the argument.
*
*/

unsigned short ParseARexx(struct RexxMsg *ARexxMessage, unsigned long **Arg) {

	unsigned short i, l;
	char *p, *q;

	ASSERT(ARexxMessage != 0);
	ASSERT(Arg != 0);

	*Arg = NULL;
	SetRexxError(ARexxMessage, 0);

	q = p = ARexxMessage->rm_Args[0];

	while(*p>' ') p++;
	l = p-q;
	if (*p) p++;

	ASSERT(l);

	for(i=0; i<sizeof(Comm)/sizeof(CommandData); i++)
		if (Strnicmp(Comm[i].Name, q, l)==0) {

			ASSERT(RDABuffer);
			ASSERT(Temp);

			l = FALSE;
			q = p;
			while(*p++) if (*p != ' ' || *p != '\t' || *p != '\n') {
				l = TRUE;
				break;
			}

			if (l) {
				ReleaseRexxArgs();
				memset(Par, 0, sizeof(Par));

				strncpy(Temp, q, TEMPBUFFERSIZE-2);
				strcat(Temp, "\n");

				RDA->RDA_Buffer = RDABuffer;
				RDA->RDA_BufSiz = RDABUFFERSIZE;
				RDA->RDA_Source.CS_Buffer = Temp;
				RDA->RDA_Source.CS_Length = strlen(Temp);
				RDA->RDA_Source.CS_CurChr = 0;
				RDA->RDA_Flags = RDAF_NOPROMPT;
				RDA->RDA_DAList = NULL;

				if (RA = ReadArgs(Comm[i].Template, Par, RDA)) {
					*Arg = Par;
					return(Comm[i].Action);
				}
			}
			else {
				*Arg = NULL;
				if (Comm[i].Flags & COMMF_NOARGS) return(Comm[i].Action);
			}
			break;
		}

	SetRexxError(ARexxMessage, 10);
	return(0);
}

void SaveConfig(void) {

	BPTR h;

	if (h = Open("PROGDIR:Startup.supdup", MODE_NEWFILE)) {
		FPuts(h, "/* SuperDuper 3.13 startup configuration file */\n");

		SaveFlagConfig(h);
		SaveWindowConfig(h);
		Close(h);
	}
	else Acknowledge(CantOpenStartupFileMsg);
}
