#include "SD.h"

#define ESC_CODE 0x1B

#define GADGETMASK	(0xFFFFBF00UL)
#define COPYMASK 		(0x3FFCC100UL)
#define READMASK		(0x3FFFC100UL)
#define WRITEMASK		(0x3FF0C100UL)
#define CHECKMASK		(0x3FFFC100UL)
#define FORMATMASK	(0x3E0CC100UL)

#define NO_BUFFER_MASK	(0xFFFFF3FFUL)
#define RAM_BUFFER_MASK	(0xFFFFFDFFUL)
#define HD_BUFFER_MASK	(0xFFFFFDFFUL)
#define VD_BUFFER_MASK	(0xFFFFFDFFUL)

#define NO_BUFFER_COPYING_MASK	(0xFFFFF3FFUL)
#define RAM_BUFFER_COPYING_MASK	(0xFBFBFDFFUL)
#define HD_BUFFER_COPYING_MASK	(0xF9FBFDFFUL)
#define VD_BUFFER_COPYING_MASK	(0xE7FFFDFFUL)


void __saveds CopyTask(void);

long	stdin, stdout;	/* for debug I/O & ARexx */
char	GUI = 1,			/* Do we have a GUI? */
		ARexx = 1,		/* This is set at startup to "do we want ARexx?".
								In case we want it, the main allocation routines
								sets it to "can the ARexx gadget work?" */
		NoStartup,		/* Should we skip the ARexx startup file? */
		Compression,	/* Are we using compression? */
		MungedBuffer,	/* Is the current buffer munged ? */
		Buffer,			/* Are we buffering? */
		StoreError,
		Requesters = 1,/* Do we want system requesters to appear? */
		RecalibrateCheck = 1,  /* Do we want to check errors while recalibrating? */
		SmartRefresh = 1,		  /* Do we want SMART_REFRESH windows on the Workbench? */
		PrintErrors;	/* Do we want printed output of disk errors? */

unsigned short Errors[4],			/* Number of errors for each unit */
					Retries[4],			/* Number of retries for each unit */
					Copies;				/* Number of buffered copies up to now */


struct ExtendedBlitNode EBN;		/* A special blitnode for all our blitter operations */

unsigned char *ReadBuffer[3], *VerifyBuffer[2]; /* "real" read and verify buffers; they're
																	normally accessed through VB[] and B[],
																	so that offsetting for SYNCs and gaps
																	is included automatically. */

struct DiskBlockServerData volatile DiskBlockServerData; /* This is where we pass data to the interrupt
																				DSKBLK server */

static unsigned char NoBufferMsg[] = "Sorry, I need a buffer.",
							CantOverlapMsg[] = "Can't overlap source and destination.",
							*NoBufferMsgs[3] = { "Not enough memory.", "Can't open the file.", "Can't open the requested device." },
							NoXpkMsg[] = "Can't open xpkmaster.library.",
							NoXpkCompMsg[] = "Can't open Xpk compressor.",
							NoDestMsg[] = "Sorry, I need a destination.",
							NoSourceMsg[] = "Sorry, I need a source.",
							NoDiskPresentMsg[] = "No disk in unit x.",
							WriteProtectedMsg[] = "Disk in unit x is write protected.",
							CantMixFloppyTypes[] = "Can't mix floppy types.",
							CantAllocateDiskBuffers[] = "Can't allocate disk buffers.",
							CantAllocateCompBuffer[] = "Can't allocate compression buffer.",
							ChecksumErrorMsg[] = "Checksum error: buffer munged.",
							InvalidBufferMsg[] = "The buffer is not valid.",
							ErrorsMsg[] = "There were errors.",
							WrongTrackNumbersMsg[] = "Track range not valid.",
							ProtectSourceMsg[] = "Better write-protect your sources.",
							RangeMismatchMsg[] = "No buffer data in this range.",
							NoCompressionMsg[] = "Compression not enabled.",
							CompressionErrorMsg[] = "Compression error.",
							IFFErrorMsg[] = "IFF syntax error.",
							BufferMungedMsg[] = "A track buffer has been munged.",
							BufferIOErrorMsg[] = "Buffer I/O error.",
							StartupName[] = "Startup.supdup",
							GfxName[] = "graphics.library",
							UNKNOWN[] = "<UNKNOWN>",
							NOTDOS[] = "<NDOS>",
							CopyTaskName[] = "SD_Copier";

static struct EClockVal StartECLock; /* for timing */

struct Resource *DiskBase;
struct DiscResourceUnit DRU = {
	{ { NULL, NULL, NT_MESSAGE, 127, "SD_DRU" }, NULL, sizeof(struct DiscResourceUnit)},
	{ { NULL, NULL, NT_INTERRUPT, 10, NULL }, (APTR)&DiskBlockServerData, InterruptCode },
	{ { NULL, NULL, NT_INTERRUPT, 10, NULL }, (APTR)&SyncIntData, SyncCode },
	{ { NULL, NULL, NT_INTERRUPT, 0, NULL }, NULL, NULL }
};

static unsigned char CIA,					/* For holding CIA registers values */
							Dest,					/* Current destination mask (from gadgets) */
							ActualDest,			/* "Real" destination mask in the current operation */
							ActualSource,		/* "Real" source number for the current copy */
							Verify = 1,			/* Is verify on? */
							Auto,					/* Is auto on? */
							Date = 1,			/* Is date adaption on? */
							UseFFS = 1,			/* Is FFS on? */
							UseIntlFS = 1,		/* Is international FS on? */
							UseDirCache,		/* Is directory cache FS on? */
							Copying,				/* What are we doing? */
							GapOpt,				/* Is the gap verify optimization on? */
							RetryNumber = 4,	/* How many retries? */
							DestNumber,			/* How many destination units? */
							DestDrive[4],		/* What are the destination drives? */
							DrawColor[4],		/* In what color are we drawing the bar? */
							ValidBuffer,		/* Is the current buffer valid? */
							AutoFlag,			/* Should we autostart ? */
							IsDos,				/* Is the current disk a DOS disk? */
							BufferIsDos,		/* Is the current buffer a DOS disk? */
							Unknown,				/* Is the name of the current disk unknown? */
							Modulo,				/* What's the modulo for accessing buffers? */
							PresentDisks,		/* What disks are linked to the system? */
							DisksOut,			/* What disks have been pulled out (at least 1 time)? */
							DisksIn,				/* What disks are in? */
							IncName,				/* Should we increment the format label? */
							MultipleRootBlocks,	/* Do we have multiple root blocks? (e.g., for date adaption) */
							Stopped;				/* Set by the copy task if the copy operation was stopped */

static unsigned char volatile Done;		/* Set this to one when everything is over */

static signed char	Source = -1,	/* The number of the source */
							Offset[3];		/* What offset are we using when accessing a buffer? */

static unsigned short	StartTrack,			/* First track to copy */
								EndTrack = 159,	/* Last Track to copy */
								ST = 0,				/* Selected start track */
								ET = 159,			/* Selected end track */
								BST,					/* Buffer start track */
								BET,					/* Buffer end track */
								FailedGaps,			/* How many gap optimization have failed? */
								Gap[4],				/* How long is the gap? */
								WriteGap,			/* Write gap length */
								PendingMacros;

static unsigned char *VB[2], *B[5];  /*	The buffers, offset in such a way
														you can set SYNCs and inter-sector
														gaps *after* reading in */

/*
 * Masks for EnableGadgets. They specify what button gadgets should be enabled
 * during an operation. The LSB is the df0: source gadget, and so on.
 */

static unsigned long GadgetMask,
							CurrentMask,
							TypeMask[4] = { NO_BUFFER_MASK, RAM_BUFFER_MASK, HD_BUFFER_MASK, VD_BUFFER_MASK },
							CopyingTypeMask[4] = { NO_BUFFER_COPYING_MASK, RAM_BUFFER_COPYING_MASK, HD_BUFFER_COPYING_MASK, VD_BUFFER_COPYING_MASK },
							ClickCount,		/* How many time click have passed? */
							Signals;			/* The signals we get in the main loop */

unsigned long	StartMask,
					StopMask,
					CopierMask;						/* Masks for copy task <-> main task communication */

static struct RexxMsg	*PendingRexxMsg,	/* Do we have a Rexx message pending? */
								*RexxMsg,			/* To read the ARexx port */
								*RemRexx;			/* To remember if something was read from the ARexx port */


struct ExecBase *SysBase;
struct WBStartup *_WBenchMsg;

LONG main(void);

LONG __saveds StdWrapper(void) {

	struct Process *Me;
	long rc;
	BPTR curdir;

	SysBase = *((void **)4);
	if (!(Me = (struct Process *)FindTask(NULL))->pr_CLI) {
		WaitPort(&Me->pr_MsgPort);
		_WBenchMsg = (struct WBStartup *)GetMsg(&Me->pr_MsgPort);
	}
	if (DOSBase = (void *)OpenLibrary("dos.library", 37)) {
		if (_WBenchMsg && (curdir = DupLock(_WBenchMsg->sm_ArgList[0].wa_Lock))) CurrentDir(curdir);
		rc = main();
		if (_WBenchMsg && curdir) UnLock(curdir);
		CloseLibrary(DOSBase);
	}
	if (_WBenchMsg) {
		Forbid();
		ReplyMsg(_WBenchMsg);
	}
	return(rc);
}



void FlushVars(void) {}


void SaveFlagConfig(BPTR h) {

	int i;

	FPrintf(h, REQUESTERS_COMMAND" %s\n", Requesters ? "On" : "Off");
	FPrintf(h, RECALIBRATECHECK_COMMAND" %s\n", RecalibrateCheck ? "On" : "Off");
	FPrintf(h, SMARTREFRESH_COMMAND" %s\n", SmartRefresh ? "On" : "Off");

	if (Dest) {
		FPrintf(h, DESTINATION_COMMAND);
		for(i=0; i<4; i++) if (MASK(i) & Dest) FPrintf(h, " %ld", (ULONG)i);
	}
	else FPrintf(h, DESTINATION_COMMAND" Off");
	FPrintf(h, "\n");

	if (Source>-1) FPrintf(h, SOURCE_COMMAND" %ld\n", (ULONG)Source);
	else FPrintf(h, SOURCE_COMMAND" Off\n");

	FPrintf(h, COPYMODE_COMMAND" %s\n", Buffer == 0 ? "Disk2Disk" :
													(Buffer == RAM_BUFFER ? "Buffer" :
													(Buffer == HD_BUFFER ? "HDBuffer" : "VDBuffer")));
}


/* Here we reset all vars related to automatic operation start */

static void ResetAuto(void) {
	DisksOut = 0;
	AutoFlag = 0;
}

/* Here we stop all motor we are using */

static void StopMotorsInUse(void) {
	MotorOff(ActualDest | MASK(ActualSource));
}

/* Here we start all motors we are using */

static void StartMotorsInUse(void) {
	MotorOn(ActualDest | MASK(ActualSource));
}

/* This function is called after any copy operation. It a) tells (if possible)
	the user that the operation is complete, b) enables the gadgets,
	c) if there is no ARexx message pending displays an error requester if it's
	the case, otherwise sets the return codes. Copying is set to 0, and a
	ResetAuto() is performed. This function has to be called from the main task. */

static void AfterCopy(void) {

	register unsigned char i;

	if (!Stopped && !MungedBuffer && !StoreError) {
		if (Copying == WRITE || Copying == READ) Say("Pass complete.");
		else Say("Operation complete.");
	}

	WriteStatus("Idle");

	if (MungedBuffer) Acknowledge(BufferMungedMsg);

	if (StoreError) switch(StoreError) {
		case ERR_CHECKSUM:
			Acknowledge(ChecksumErrorMsg);
			break;
		case ERR_NOCOMP:
			Acknowledge(NoCompressionMsg);
			break;
		case ERR_IO:
			Acknowledge(BufferIOErrorMsg);
			break;
		case ERR_CANTOPENVDISK:
			Acknowledge(NoBufferMsgs[VDISK_BUFFER-1]);
			break;
		case ERR_IFF:
			Acknowledge(IFFErrorMsg);
			break;
		case ERR_CANTOPENFILE:
			Acknowledge(NoBufferMsgs[HD_BUFFER-1]);
			break;
		case ERR_COMP:
			Acknowledge(CompressionErrorMsg);
			break;
		case ERR_WRONGTYPE:
			Acknowledge(CantMixFloppyTypes);
			break;
	}

	EnableGadgets(CurrentMask);
	for(i=0; i<4; i++) {
		if (((MASK(i) & ActualDest) || i == ActualSource) && Errors[i]) {
			Acknowledge(ErrorsMsg);
			break;
		}
	}
	if (PendingRexxMsg) {
		if (MungedBuffer || StoreError) SetRexxError(PendingRexxMsg, 20);
		else if (i<4) SetRexxError(PendingRexxMsg, (Compression && BET != ET) ? 9 : 8);
		else if (Compression && BET != ET) SetRexxError(PendingRexxMsg, 2);
		ReplyMsg((void *)PendingRexxMsg);
		PendingRexxMsg = NULL;
	}
	StoreError = MungedBuffer = Stopped = FALSE;
	ResetAuto();
	Copying = 0;
	UpdateProgress(FALSE);
	UpdateMainProgress(FALSE);
	UpdateElapsed();
}

/*	Here we stop any operation. If we weren't copying, and the buffer is on,
	and it's not valid OR it's partial, we wipe the progress bar and reset
	the pass counter. You can call this function from everywhere. */

static void Stop(void) {

	if (Copying) Signal(Copier, StopMask);
	else {
		if (Buffer && (!ValidBuffer || BET<GetECyl()*2+1 || BST>GetSCyl()*2)) {
			WipeProgress(0);
			ValidBuffer = FALSE;
			if (Buffer == RAM_BUFFER) FreeBuffers[RAM_BUFFER]();
			BET = 159;
		}
	}
}


/* Here we try to get a buffer. */

static void GetBuffer(unsigned char Type) {

	FreeBuffers[Buffer]();
	Buffer = NO_BUFFER;

	if (Type) {
		if (!ReAllocBuffers[Type]()) {
			Acknowledge(NoBufferMsgs[Type-1]);
			SetRexxError(RexxMsg, 5);
		}
		else Buffer = Type;
	}

	BST = 0;
	BET = 159;
	BufferIsDos = Copies = ValidBuffer = 0;
	WriteCopies();
	ResetAuto();
}

/* Here we draw the dragbar, basing on the track number, and we update
	the copy time. This has to be called from the copy task. */

static void DrawMainBar(unsigned short Tr) {
	if (Tr%2) {
		DrawMainProgress(Tr/2, 3);
		WriteElapsed(&StartECLock);
	}
}

static void DrawBar(unsigned char Unit, unsigned short Tr) {
	if (Tr%2) DrawProgress(Unit, Tr/2, DrawColor[Unit]);
}

/* Here we verify a track and in case we retry it */

static void VerifyAndRetry(unsigned char Unit, unsigned short Tr, unsigned char *VBuffer) {

	unsigned char i = 0;

	if (VerifyTrack(B[Tr%Modulo]+Offset[Tr%Modulo], VBuffer, Unit, Tr)) {
		if (RetryNumber) {
			MoveAndBackupTrack(Unit, Tr);
			WriteRetriesErrors(Unit, Tr, FALSE);
			do {
				if (i++ == RetryNumber) break;
				Retries[Unit]++;
				StartWrite(MASK(Unit), Tr, B[Tr%Modulo]+Offset[Tr%Modulo]-WriteGap, WriteLength+WriteGap, Unit, VBuffer, TRUE);
				WaitDMA();
			} while(VerifyTrack(B[Tr%Modulo]+Offset[Tr%Modulo], VBuffer, Unit, Tr));
			RestoreTrack(Unit);
		}
		else i = 1;

		if (i == RetryNumber+1) {
			Retries[Unit]-=RetryNumber;
			Errors[Unit]++;
			WriteRetriesErrors(Unit, Tr, TRUE);
			DisplayBeep(NULL);
		}
	}
}

/* Here we recode a track just read for writing. If something fails, we
	retry it. */

static void RecodeAndRetry(unsigned short Tr) {

	unsigned char i = 0;

	Offset[Tr%Modulo] = FindOffset(B[Tr%Modulo]);
	if (RecodeTrack(B[Tr%Modulo]+Offset[Tr%Modulo], ActualSource, Tr)) {
		if (RetryNumber) {
			WriteRetriesErrors(ActualSource, Tr, FALSE);
			MoveAndBackupTrack(ActualSource, Tr);
			do {
				if (i++ == RetryNumber) break;
				Retries[ActualSource]++;
				StartRead(ActualSource, Tr, B[Tr%Modulo], FALSE);
				WaitDMA();
				Offset[Tr%Modulo] = FindOffset(B[Tr%Modulo]);
			} while(RecodeTrack(B[Tr%Modulo]+Offset[Tr%Modulo], ActualSource, Tr));
			RestoreTrack(ActualSource);
		}
		else i = 1;

		if (i == RetryNumber+1) {
			Retries[ActualSource]-=RetryNumber;
			Errors[ActualSource]++;
			WriteRetriesErrors(ActualSource, Tr, TRUE);
			DisplayBeep(NULL);
		}
	}
}

/* Here we store a track just read, and in case we retry. */

static void StoreAndRetry(unsigned short Tr) {

	unsigned char i = 0;

	if (StoreTrack(B[Tr%5], ActualSource, Tr)) {
		if (RetryNumber) {
			WriteRetriesErrors(ActualSource, Tr, FALSE);
			MoveAndBackupTrack(ActualSource, Tr);
			do {
				if (i++ == RetryNumber) break;
				Retries[ActualSource]++;
				StartRead(ActualSource, Tr, B[Tr%5], FALSE);
				WaitDMA();
			} while(StoreTrack(B[Tr%5], ActualSource, Tr));
			RestoreTrack(ActualSource);
		}
		else i = 1;

		if (i == RetryNumber+1) {
			DisplayBeep(NULL);
			Retries[ActualSource]-=RetryNumber;
			Errors[ActualSource]++;
			WriteRetriesErrors(ActualSource, Tr, TRUE);
		}
	}
}


/* Here we get a track from the buffer, and if the checksum is wrong we stop
	the copy. */

static void RetrieveAndCheck(unsigned short Tr) {

	RetrieveTrack(Tr, B[Tr%Modulo]);
}
/* Here we do the action corresponding to an option gadget */

static void SwitchNumber(unsigned char GadgetNumber, unsigned short i) {

	switch(GadgetNumber) {
		case VERIFY:	Verify = i;
							break;
		case DATE:		Date = i;
							break;
		case FFS:		UseFFS = i;
							break;
		case INTL:		UseIntlFS = i;
							break;
		case DIRCACHE:	UseDirCache = i;
							break;
		case AUTO:		Auto = i;
							ResetAuto();
							if (i && (PresentDisks & 0xF) == 1 && !RexxMsg) Acknowledge(ProtectSourceMsg);
							break;
		case INCNAME:	IncName = i;
							break;
		case PRINTERRORS:
							if (PrintErrors = i) OpenErrorConsole();
							else CloseErrorConsole();
							break;
		case TALK:		if (i) {
								if (!OpenVoice()) {
									SelectGadget(TALK, FALSE);
									SetRexxError(RexxMsg, 5);
								}
							}
							else CloseVoice();
							break;
		case COMPRESSION:
							if (Compression == i) break;
							if (i) {
								char LibName[64];

								strcat(strncat(strcpy(LibName, "libs:compressors/xpk"), GetXPKLibName(), 4), ".library");

								if (XpkBase = OpenLibrary("xpkmaster.library", 2)) {
									if (XpkComp = OpenLibrary(LibName, 0)) {
										if (i != Compression) ValidBuffer = 0;
										Compression = TRUE;
									}
									else Acknowledge(NoXpkCompMsg);
								}
								else Acknowledge(NoXpkMsg);

								if (!XpkComp) {
									CloseLibrary(XpkBase);
									XpkBase = NULL;
									Compression = FALSE;
									SelectGadget(COMPRESSION, FALSE);
								}
							}
							else {
								CloseLibrary(XpkBase);
								XpkBase = NULL;
								CloseLibrary(XpkComp);
								XpkComp = NULL;
								FreeCompBuffer();
								if (i != Compression) ValidBuffer = 0;
								Compression = FALSE;
							}
							break;

	}
}


/* Reset the copy state to an unknown disk */

static void ResetToUnknown(void) {
	IsDos = 0;
	if (!Unknown) AddName(UNKNOWN);
	Unknown = 1;
}

/* Here we take the action corresponding to an action gadget */

static unsigned char SwitchActions(short GN) {

	unsigned short i;
	ULONG Type;

	if (Copying) return(1);
	ResetAuto();

	if ((i = GetSCyl()*2) != ST) WipeProgress(0);
	ST = i;
	if ((i = GetECyl()*2+1) != ET) WipeProgress(0);
	ET = i;

	if (ET<ST || ET>159) {
		Acknowledge(WrongTrackNumbersMsg);
		return(7);
	}
	if ((GN == COPY || GN == READ || GN == CHECK) && Source<0) {
		Acknowledge(NoSourceMsg);
		return(3);
	}
	if (GN == COPY && INDEST(Source)) {
		Acknowledge(CantOverlapMsg);
		return(3);
	}
	if ((GN == READ || GN == WRITE) && !Buffer ) {
		Acknowledge(NoBufferMsg);
		return(3);
	}
	if ((GN == WRITE || GN == COPY || GN == FORMAT) && !Dest) {
		Acknowledge(NoDestMsg);
		return(3);
	}
	if (GN == WRITE && Buffer == RAM_BUFFER) {
		if (ValidBuffer) {
			if (ST>BET || ET<BST) {
				Acknowledge(RangeMismatchMsg);
				return(7);
			}
		}
		else {
			Acknowledge(InvalidBufferMsg);
			return(4);
		}
	}
	ActualDest = Dest;
	ActualSource = Source;
	if ((Copying = GN) == READ || Copying == CHECK) ActualDest = MASK(4);
	else if (Copying == WRITE || Copying == FORMAT) ActualSource = 4;

	FailedGaps = 0;
	GapOpt = 1;
	WriteGap = GapSize;

	Type = DRT_EMPTY;

	for(i=0; i<4; i++) {
		Errors[i] = Retries[i] = 0;
		if (INACTUALDEST(i) || i==ActualSource) {
			CIA = GetReallyCIAAPRA(i);
			if (!(CIA & CIAF_DSKCHANGE)) {
				NoDiskPresentMsg[16] = 48+i;
				Acknowledge(NoDiskPresentMsg);
				Copying = 0;
				return(5);
			}
			if (Type == DRT_EMPTY) Type = ReadUnitID(i);
			else if (Type != ReadUnitID(i)) {
				Acknowledge(CantMixFloppyTypes);
				Copying = 0;
				return(6);
			}
			if (INACTUALDEST(i) && !(CIA & CIAF_DSKPROT)) {
				WriteProtectedMsg[13] = 48+i;
				Acknowledge(WriteProtectedMsg);
				Copying = 0;
				return(6);
			}
			DisksIn |= MASK(i);
		}
	}

	if (Type != CurrentType) {
		if (Copying == WRITE && Buffer != HD_BUFFER) {
			Acknowledge(CantMixFloppyTypes);
			Copying = 0;
			return(6);
		}
	}

	CurrentType = Type;

	if (Compression && !ReAllocCompBuffer()){
		Acknowledge(CantAllocateCompBuffer);
		Copying = 0;
		return(20);
	}
	if (!ReAllocTDBuffers()) {
		Acknowledge(CantAllocateDiskBuffers);
		Copying = 0;
		return(20);
	}

	B[3] = VB[0] = VerifyBuffer[0]+8;
	B[4] = VB[1] = VerifyBuffer[1]+8;

	if (Verify) DrawColor[0] = DrawColor[1] = DrawColor[2] = DrawColor[3] = 3;
	else DrawColor[0] = DrawColor[1] = DrawColor[2] = DrawColor[3] = 2;
	Modulo = 3;
	StartTrack = ST;
	EndTrack = ET;
	for(i=0; i<3; i++) B[i] = ReadBuffer[i]+GapSize+8;

	if (Copying == READ || Copying == CHECK) {
		DrawColor[Source] = 1;
		if (Copying == CHECK) ResetToUnknown();
		else {
			if (BET >= ET) {
				BST = StartTrack;
				ResetToUnknown();
				BufferIsDos = 0;
			}
			else BST = StartTrack = max(ST, BET+1);
			BET = EndTrack;
		}
		Modulo = 5;
	}
	else if (Copying == COPY) {
		DrawColor[Source] = 1;
		ResetToUnknown();
	}
	else {
		if (Copying == FORMAT) {
			IsDos = 1;
		}
		else {
			IsDos = BufferIsDos;
			StartTrack = max(BST, ST);
			EndTrack = min(ET, BET);
		}
		for(i=0; i<3; i++) {
			B[i] = ReadBuffer[i]+GapSize;
			Offset[i] = 0;
		}
	}

	DestNumber = 0;
	for(i=0; i<4; i++)
		if (INACTUALDEST(i)) {
			DestDrive[DestNumber++] = i;
		}

	ReadEClock(&StartECLock);
	StartMotorsInUse();
	SetUpDisksA();

	switch(Copying) {
		case READ:
			EnableGadgets(CurrentMask & CopyingTypeMask[Buffer] & READMASK);
			WriteStatus("Read");
			break;
		case CHECK:
			EnableGadgets(CurrentMask & CHECKMASK);
			WriteStatus("Check");
			break;
		case FORMAT:
			EnableGadgets(CurrentMask & FORMATMASK);
			WriteStatus("Format");
			break;
		case WRITE:
			EnableGadgets(CurrentMask & CopyingTypeMask[Buffer] & WRITEMASK);
			WriteStatus("Write");
			break;
		case COPY:
			EnableGadgets(CurrentMask & CopyingTypeMask[Buffer] & COPYMASK);
			WriteStatus("Copy");
			break;
	}

	WipeProgress(StartTrack/2);
	SetUpDisksB(ActualDest | MASK(ActualSource), StartTrack);

	MultipleRootBlocks = Date || (IncName && Copying == FORMAT);

	Signal(Copier, StartMask);

	return(0);
}

void __saveds CopyTask(void) {

	register unsigned int i;
	register unsigned short Track;
	BPTR CDLock;

	DiskBlockServerData.DBSD_ISD.ISD_Task = SoftIntData.SID_ISD.ISD_Task = EBN.EBN_Task = Copier = FindTask(NULL);
	Done = !(CopyTaskAllocStuff() && CopyTaskOpenTimer());
	CDLock = CurrentDir(DupLock(((struct Process *)Me)->pr_CurrentDir));
	Signal(Me, CopierMask);

	FOREVER {
		Wait(StartMask);
		if (Done) break;
		SetSignal(0, StopMask);
		FlushVars();

		Track = StartTrack;

		switch(Copying) {
			case READ:
			case CHECK:
				StartRead(ActualSource, Track, B[Track%5], FALSE);
				break;
			case COPY:
				StartRead(ActualSource, Track, B[Track%3], FALSE);
				MoveHead(ActualSource, Track+1);
				break;
			case FORMAT:
				CreateTrack(Track, B[Track%3], UseFFS, UseIntlFS, UseDirCache);
				break;
			case WRITE:
				StartRetrieve(Track, B[Track%3], B[(Track+1)%3]);
				WaitStore(FALSE);
				if (!StoreError) {
					StartRetrieve(Track, NULL, NULL);
					RetrieveAndCheck(Track);
				}
		}
		Track++;

		FOREVER {
			if (MungedBuffer || StoreError || (SetSignal(0,0) & StopMask)) {
				DisplayBeep(NULL);
				Stopped = TRUE;
				break;
			}
			if (Track > 1+EndTrack) break;
			RetryNumber = GetRetryNumber();
			if (Copying == READ || Copying == CHECK) {
				if (Track<=EndTrack) {
					MoveHead(ActualSource, Track);
					StartRead(ActualSource, Track, B[Track%5], FALSE);
				}
				else WaitDMA();
				DrawBar(ActualSource, Track);
				DrawMainBar(Track);
				StoreAndRetry(Track-1);
				if (Copying == READ) {
					if (!(Track%2)) {
						if (WaitStore(FALSE)) {
							EndTrack = Track-5;
							WipeProgress((EndTrack+1)/2);
							continue;
						}
						StartStore(Track-2);
					}
				}
				if (Track == 81 && IsDos) {
					ChangeName(GetNameBuffered());
					Unknown = 0;
				}
				if (Track == 1 && !(BufferIsDos = IsDos = IsDosBuffered())) {
					ChangeName(NOTDOS);
					Unknown = 0;
				}
			}
			else if (Copying == COPY) {
				if (Verify)	{
					if (Track<=EndTrack) {
						StartRead(ActualSource, Track, B[Track%3], FALSE);
						DrawBar(ActualSource, Track+1-(Track == StartTrack+1));
						for(i=0; i<DestNumber; i++) DrawBar(DestDrive[i], Track+1-(Track == StartTrack+1));
						DrawMainBar(Track+1-(Track == StartTrack+1));
					}
					if (Track == StartTrack+1) {
						RecodeAndRetry(StartTrack);
						if (StartTrack == 0 && !(IsDos = IsDosMFM(B[0]+Offset[0]))) {
							ChangeName(NOTDOS);
							Unknown = 0;
						}
					}
					else {
						VerifyAndRetry(DestDrive[DestNumber-1], Track-2, VB[1-DestNumber%2]);
						if (Track == StartTrack+2) {
							Gap[DestDrive[DestNumber-1]] = GapLength(VB[1-DestNumber%2]);
							WriteGap = 0;
							for(i=0; i<DestNumber; i++) if (Gap[DestDrive[i]]>WriteGap) WriteGap = Gap[DestDrive[i]];
						}
					}
					if (Track == 81 && Date && IsDos) UpdateMFMRootBlock(B[80%3]+Offset[80%3], TRUE, FALSE);
					if (Track == 81 && IsDos) {
						ChangeName(GetNameMFM(B[80%3]+Offset[80%3]));
						Unknown = 0;
					}
					if (Track<EndTrack) MoveHead(ActualSource, Track+1);
					if (Track != 81 || !Date || !IsDos) StartWrite(ActualDest, Track-1, B[(Track-1)%3]+Offset[(Track-1)%3]-WriteGap, WriteLength+WriteGap, DestDrive[0], VB[0], Track == StartTrack+1 || !GapOpt);
					else {
						StartWrite(MASK(DestDrive[0]), 80, B[80%3]+Offset[80%3]-WriteGap, WriteLength+WriteGap, DestDrive[0], VB[0], StartTrack == 80 || !GapOpt);
					}
					if (Track<=EndTrack) RecodeAndRetry(Track);
					WaitDMA();
					if (GapOpt && (char)(DecodeLong(VB[0]+FindOffset(VB[0])+SE_Header)) != SPT) {
						StartRead(DestDrive[0], Track-1, VB[0], Track == StartTrack+1);
						FailedGaps++;
						if (FailedGaps>(Track-StartTrack)/10+1) GapOpt = 0;
						WaitDMA();
					}
					if (Track<=EndTrack) MoveHead(DestDrive[0], Track);
					for (i=1; i<DestNumber; i++) {
						if (Track != 81 || !Date || !IsDos) StartRead(DestDrive[i], Track-1, VB[i%2], Track == StartTrack+1);
						VerifyAndRetry(DestDrive[i-1], Track-1, VB[1-i%2]);
						if (Track == StartTrack+1) Gap[DestDrive[i-1]] = GapLength(VB[1-i%2]);
						if (Track == 81 && Date && IsDos) {
							UpdateMFMRootBlock(B[80%3]+Offset[80%3], TRUE, FALSE);
							StartWrite(MASK(DestDrive[i]), Track-1, B[80%3]+Offset[80%3]-WriteGap, WriteLength+WriteGap, DestDrive[i], VB[i%2], TRUE);
						}
						WaitDMA();
						if (Track<=EndTrack) MoveHead(DestDrive[i], Track);
					}
					if (Track>EndTrack) VerifyAndRetry(DestDrive[DestNumber-1], EndTrack, VB[1-DestNumber%2]);
				}
				else {
					if (Track<=EndTrack) {
						StartRead(ActualSource, Track, B[Track%3], FALSE);
						DrawBar(ActualSource, Track+1-(Track == StartTrack+1));
						for(i=0; i<DestNumber; i++) DrawBar(DestDrive[i], Track+1-(Track == StartTrack+1));
						DrawMainBar(Track+1-(Track == StartTrack+1));
					}
					if (Track == StartTrack+1) {
						RecodeAndRetry(StartTrack);
						if (StartTrack == 0 && !(IsDos = IsDosMFM(B[0]+Offset[0]))) {
							ChangeName(NOTDOS);
							Unknown = 0;
						}
					}
					if (Track == 81 && IsDos) {
						ChangeName(GetNameMFM(B[80%3]+Offset[80%3]));
						Unknown = 0;
					}
					if (Track<EndTrack) MoveHead(ActualSource, Track+1);
					if (Track != 81 || !Date || !IsDos) StartWrite(ActualDest, Track-1, B[(Track-1)%3]+Offset[(Track-1)%3]-WriteGap, WriteLength+WriteGap, DestDrive[0], (Track == StartTrack+1) ? VB[0] : NULL, TRUE);
					else for(i=0; i<DestNumber; i++) {
						WaitDMA();
						UpdateMFMRootBlock(B[(Track-1)%3]+Offset[(Track-1)%3], TRUE, FALSE);
						StartWrite(MASK(DestDrive[i]), Track-1, B[(Track-1)%3]+Offset[(Track-1)%3]-WriteGap, WriteLength+WriteGap, DestDrive[0], (i == 0 && Track == StartTrack+1) ? VB[0] : NULL, TRUE);
					}
					if (Track == StartTrack+1) {
						for(i=1; i<DestNumber; i++) {
							StartRead(DestDrive[i], Track-1, VB[i%2], TRUE);
							Gap[DestDrive[i-1]] = GapLength(VB[(i-1)%2]);
						}
						WaitDMA();
						Gap[DestDrive[i-1]] = GapLength(VB[(i-1)%2]);
						WriteGap = 0;
						for(i=0; i<DestNumber; i++) if (Gap[DestDrive[i]]>WriteGap) WriteGap = Gap[DestDrive[i]];
					}
					if (Track<=EndTrack) {
						RecodeAndRetry(Track);
						MoveHeads(ActualDest, Track);
					}
				}
			}
			else {
				if (Verify) {
					if (Track != 81 || !MultipleRootBlocks || !IsDos) StartWrite(ActualDest, Track-1, B[(Track-1)%3]-WriteGap, WriteLength+WriteGap, DestDrive[0], VB[DestNumber%2 ? Track%2 : 0], Track == StartTrack+1 || !GapOpt);
					else {
						UpdateMFMRootBlock(B[80%3], Date, IncName);
						StartWrite(MASK(DestDrive[0]), 80, B[80%3]-WriteGap, WriteLength+WriteGap, DestDrive[0], VB[DestNumber%2], StartTrack == 80 || !GapOpt);
					}
					for(i=0; i<DestNumber; i++) DrawBar(DestDrive[i], Track);
					DrawMainBar(Track);
					if (Track>StartTrack+1) {
						VerifyAndRetry(DestDrive[DestNumber-1], Track-2, VB[DestNumber%2 ? (Track+1)%2 : 1]);
						if (Track == StartTrack+2) {
							Gap[DestDrive[DestNumber-1]] = GapLength(VB[DestNumber%2 ? (Track+1)%2 : 1]);
							WriteGap = 0;
							for(i=0; i<DestNumber; i++) if (Gap[DestDrive[i]]>WriteGap) WriteGap = Gap[DestDrive[i]];
						}
					}
					if (Track<=EndTrack) {
						if (Copying == FORMAT) CreateTrack(Track, B[Track%3], UseFFS, UseIntlFS, UseDirCache);
						else {
							if (Track%2) {
								if (WaitStore(Track == EndTrack)) break;
								if (Track != EndTrack) StartRetrieve(Track+1, B[(Track+1)%3], B[(Track+2)%3]);
							}
							else {
								if (WaitStore(FALSE)) break;
								StartRetrieve(Track, NULL, NULL);
							}
							RetrieveAndCheck(Track);
						}
					}
					WaitDMA();
					if (GapOpt && (char)(DecodeLong(VB[DestNumber%2 ? Track%2 : 0]+FindOffset(VB[DestNumber%2 ? Track%2 : 0])+SE_Header)) != SPT) {
						StartRead(DestDrive[0], Track-1, VB[DestNumber%2 ? Track%2 : 0], Track == StartTrack+1);
						FailedGaps++;
						if (FailedGaps>(Track-StartTrack)/10+1) GapOpt = 0;
						WaitDMA();
					}
					if (Track<=EndTrack) MoveHead(DestDrive[0], Track);
					for (i=1; i<DestNumber; i++) {
						if (Track != 81 || !MultipleRootBlocks || !IsDos) StartRead(DestDrive[i], Track-1, VB[DestNumber%2 ? (i+Track)%2 : i%2], Track == StartTrack+1);
						if (Track == StartTrack+1) Gap[DestDrive[i-1]] = GapLength(VB[DestNumber%2 ? (i+Track+1)%2 : (i-1)%2]);
						VerifyAndRetry(DestDrive[i-1], Track-1, VB[DestNumber%2 ? (i+Track+1)%2 : (i-1)%2]);
						if (Track == 81 && MultipleRootBlocks && IsDos) {
							UpdateMFMRootBlock(B[80%3], Date, IncName);
							StartWrite(MASK(DestDrive[i]), 80, B[80%3]-WriteGap, WriteLength+WriteGap, DestDrive[i], VB[DestNumber%2 ? (i+1)%2 : i%2], TRUE);
						}
						WaitDMA();
						if (Track<=EndTrack) MoveHead(DestDrive[i], Track);
					}
					if (Track>EndTrack) VerifyAndRetry(DestDrive[DestNumber-1], EndTrack, VB[DestNumber%2 ? Track%2 : 1]);
				}
				else {
					if (Track != 81 || !MultipleRootBlocks || !IsDos) StartWrite(ActualDest, Track-1, B[(Track-1)%3]-WriteGap, WriteLength+WriteGap, DestDrive[0], (Track == StartTrack+1) ? VB[0] : NULL, TRUE);
					else for(i=0; i<DestNumber; i++) {
						WaitDMA();
						UpdateMFMRootBlock(B[80%3], Date, IncName);
						StartWrite(MASK(DestDrive[i]), 80, B[80%3]-WriteGap, WriteLength+WriteGap, DestDrive[0], (i == 0 && Track == StartTrack+1) ? VB[0] : NULL, TRUE);
					}
					for(i=0; i<DestNumber; i++) DrawBar(DestDrive[i], Track);
					DrawMainBar(Track);
					if (Track<=EndTrack) {
						if (Copying == FORMAT) CreateTrack(Track, B[Track%3], UseFFS, UseIntlFS, UseDirCache);
						else {
							if (Track%2) {
								if (WaitStore(Track == EndTrack)) break;
								if (Track != EndTrack) StartRetrieve(Track+1, B[(Track+1)%3], B[(Track+2)%3]);
							}
							else {
								if (WaitStore(FALSE)) break;
								StartRetrieve(Track, NULL, NULL);
							}
							RetrieveAndCheck(Track);
						}
					}
					if (Track == StartTrack+1) {
						for(i=1; i<DestNumber; i++) {
							StartRead(DestDrive[i], Track-1, VB[i%2], TRUE);
							Gap[DestDrive[i-1]] = GapLength(VB[(i-1)%2]);
						}
						WaitDMA();
						Gap[DestDrive[i-1]] = GapLength(VB[(i-1)%2]);
						WriteGap = 0;
						for(i=0; i<DestNumber; i++) if (Gap[DestDrive[i]]>WriteGap) WriteGap = Gap[DestDrive[i]];
					}
					if (Track<=EndTrack) MoveHeads(ActualDest, Track);
				}
				if (Track == 1 && (Buffer == VDISK_BUFFER || Buffer == HD_BUFFER) && Copying == WRITE) BufferIsDos = IsDos = IsDosBuffered();
			}
			Track++;
		}

		WaitDMA();
		WaitHeads();
		StopMotorsInUse();
		WriteElapsed(&StartECLock);

		if (!Stopped && Copying == READ) {

			if (WaitStore(TRUE)) {
				BET = EndTrack-2;
				WipeProgress(EndTrack/2);
			}
			else BET = EndTrack;

			ValidBuffer = TRUE;

			if (StartTrack == BST) {
				Copies = 0;
				WriteCopies();
			}
		}

		if (Stopped && Copying == READ) {
			ValidBuffer = FALSE;
			WipeProgress(StartTrack/2);
			BET = BST-1;
		}

		if (!Stopped) {
			DisplayBeep(NULL);
			if (Copying == WRITE && StartTrack == BST && EndTrack == 159) {
				for(i=0; i<4; i++) if (!Errors[i] && INACTUALDEST(i)) Copies++;
				WriteCopies();
			}
		}

		WaitStore(TRUE);
		Signal(Me, CopierMask);
	}

	UnLock(CurrentDir(CDLock));
	CopyTaskCloseTimer();
	CopyTaskFreeStuff();
	Signal(Me, CopierMask);
}

/*
 * The big thing!
 * Apart from the open-all-stuff/close-all-stuff parts, this is simply a
 * preposterously huge event loop.
 */


LONG main(void) {

	register unsigned short i;
	register short j;
	unsigned char *p;
	struct Gadget *G;
	struct IntuiMessage *Msg;
	struct Message *AppMsg;
	ULONG MsgClass, Code, Qualifier;
	int WindowNumber;
	ULONG Par[OPT_DUMMYCOUNT];
	short Op, OldPri;
	unsigned char GadgetNumber;
	struct RDArgs *RA = NULL;
	struct DiskObject *DiskObj = NULL;

	InitStuff();

	memset(Par, 0, sizeof(Par));

	if (!(IconBase = OpenLibrary("icon.library", 37))) return(1);

	if (_WBenchMsg) {
		BPTR OldCD;

		OldCD = CurrentDir(_WBenchMsg->sm_ArgList[0].wa_Lock);
		if (DiskObj = GetDiskObject(_WBenchMsg->sm_ArgList[0].wa_Name)) {

			GUI = !FindToolType(DiskObj->do_ToolTypes, "NOGUI");
			ARexx = !FindToolType(DiskObj->do_ToolTypes, "NOAREXX");
			NoStartup = (FindToolType(DiskObj->do_ToolTypes, "NOSTARTUP") != 0);
			PubScreen = FindToolType(DiskObj->do_ToolTypes, "PUBSCREEN");

		}
		CurrentDir(OldCD);
	}
	else {
		if (!(RA = ReadArgs(TEMPLATE, Par, NULL))) {
			PrintFault(IoErr(), (char *)BADDR(((struct CommandLineInterface *)BADDR(((struct Process *)FindTask(NULL))->pr_CLI))->cli_CommandName)+1);
			return;
		}

		GUI = !Par[OPT_NOGUI];
		ARexx = !Par[OPT_NOAREXX];
		PubScreen = (char *)Par[OPT_PUBSCREEN];
		NoStartup = Par[OPT_NOSTARTUP];
	}


	if ((OldPri = SetTaskPri(Me = FindTask(NULL), 2))>2) SetTaskPri(Me,OldPri);

	if (!GUI && !ARexx || FindTask(CopyTaskName) || FindPort(ARexxPortName)) goto GameOver;

	if (!AllocStuff() || (DiskBase = OpenResource(DISKNAME)) == NULL ||
		!CreateNewProcTags(NP_Entry, CopyTask, NP_Name, CopyTaskName, NP_Priority, (ULONG)(Me->tc_Node.ln_Pri+1), TAG_DONE))
		goto GameOver;
	Wait(CopierMask);
	if (Done) goto GameOver;
	DRU.dru_Message.mn_ReplyPort = ResPort;
	if (!ARexxPort && !GUI) goto GameOver;

	GadgetMask = GADGETMASK | (PresentDisks = WhichDisks()) | (ARexx ? MASK(AREXX-2) : 0);

	if (!ReAllocTDBuffers()) goto GameOver;

	EnableGadgets(CurrentMask = GadgetMask & TypeMask[NO_BUFFER]);

	if (!OpenTimer() || (GUI && !SetUpWindow())) goto GameOver;

	B[3] = VB[0] = VerifyBuffer[0]+8;
	B[4] = VB[1] = VerifyBuffer[1]+8;

	if (!NoStartup && ARexxPort && StartARexxMacro(CreateArgstring(StartupName, strlen(StartupName)+1), TRUE)) PendingMacros = 1;

	PostClick();

	FOREVER {

		do {
			if (Signals & CopierMask) {
				ASSERT(Copying);
				Signals &= ~CopierMask;
				FlushVars();
				AfterCopy();
			}

			if (!Copying && CheckClick()) {
				if (Dest || Source>=0) {
					RemoveClick();
					for(i=0; i<4; i++)
						if (INMASK(i,PresentDisks)) {
							if (GetCIAAPRA(i, CIAF_DSKMOTOR, TRUE) & CIAF_DSKCHANGE) {
								DisksIn |= MASK(i);
							}
							else {
								if (DisksIn & MASK(i)) {
									TimerWait(100000);
									RecalibrateHead(i);
								}
								DisksOut |= MASK(i);
								DisksIn &= ~MASK(i);
							}
						}
					i = Dest | (Source<0 || Buffer ? 0 : MASK(Source));
					if (Dest && (Source<0 || !INDEST(Source) || Buffer) && i && (DisksIn & i) == i && (DisksOut & i) == i)
						AutoFlag = Auto;
					ClickHeads(PresentDisks, ++ClickCount);
					PostClick();
				}
			}

			if ((WindowPort && (Msg = (void *)GT_GetIMsg(WindowPort))) || AutoFlag) {

				if (Msg) {
					Code = Msg->Code;
					MsgClass = Msg->Class;
					Qualifier = Msg->Qualifier;
					G = (struct Gadget *)(Msg->IAddress);
					WindowNumber = WINDOW_NUMBER(Msg->IDCMPWindow);
					if (MsgClass == RAWKEY) Op = toupper(DeadKeyConvert(Msg));
					GT_ReplyIMsg((void *)Msg) ;
				}
				else {
					MsgClass = RAWKEY;
					if (Buffer) {
						if (Source>-1 && !(GetCIAAPRA(Source, CIAF_DSKMOTOR, TRUE) & CIAF_DSKPROT)) Op = GadgetKey[READ];
						else Op = GadgetKey[WRITE];
					}
					else if (Source>-1) Op = GadgetKey[COPY];
					else Op = GadgetKey[FORMAT];
					Code = Qualifier = 0;
					WindowNumber = WIN_MAIN;
				}

				switch(MsgClass) {
					case CLOSEWINDOW:
						if (WindowNumber == WIN_MAIN) goto GameOver;
						else CloseWin(WindowNumber);
						break;

					case REFRESHWINDOW:
						RefreshWindow(WindowNumber);
						break;

					case GADGETUP:
					case GADGETDOWN:
					case RAWKEY:
						if (MsgClass == RAWKEY) {
							if (Qualifier & IEQUALIFIER_REPEAT) break;
							GadgetNumber = 0xFFU;
							switch (Code) {
								case 0x5F:
									HyperHelp();
									break;
								case 80:
									ZoomWindow(WindowNumber, -1);
									break;
								case CURSORLEFT:
									NextWindow(WindowNumber, -1);
									break;
								case CURSORRIGHT:
									NextWindow(WindowNumber, 1);
									break;
								default:
									if (Op == 'Q' || Op == ESC_CODE) {
										if (Op == 'Q' || WindowNumber == WIN_MAIN) goto GameOver;
										else CloseWin(WindowNumber);
									}
									else if (Op) GadgetNumber = Op2Gadget(Op);
									break;
							}
							if (GadgetNumber == 0xFFU) break;
						}
						else {
							ASSERT(G)
							SetGadgetState(GadgetNumber = G->GadgetID);
						}

						ASSERT(GadgetNumber<=COPYMODE);

						if (!GadgetEnabled(GadgetNumber)) break;

						switch(GadgetNumber) {
							case COPYMODE:
										if (Copying) {
											SetCopyMode(Buffer);
											break;
										}
										if (MsgClass == RAWKEY) {
											if (Qualifier & (IEQUALIFIER_LSHIFT|IEQUALIFIER_RSHIFT))
												Code = Buffer ? Buffer-1 : BUFFER_COUNT-1;
											else Code = Buffer+1 == BUFFER_COUNT ? 0 : Buffer+1;
										}
										SetCopyMode(Code);
										GetBuffer(Code);
										SetCopyMode(Buffer);
										EnableGadgets(CurrentMask = GadgetMask & TypeMask[Buffer]);
										break;

							case COPY:
							case READ:
							case WRITE:
							case CHECK:
							case FORMAT:
										SwitchActions(GadgetNumber);
										break;

							case AREXX:	if (StartARexxMacro(GetFile(), FALSE)) PendingMacros++;
											break;

							case STOP:	Stop();
											break;

							case NOWB:	if (KillSys(-1)) goto GameOver;
											break;


							case OPTIONS:
								SetUpOptWindow();
								break;

							case INFO:
								SetUpInfoWindow();
								break;

							case ICONIFY:
								Iconify(TRUE);
								break;

							case SAVECONF:
								SaveConfig();
								break;

							case VERIFY:
							case COMPRESSION:
							case DATE:
							case PRINTERRORS:
							case INCNAME:
							case FFS:
							case INTL:
							case DIRCACHE:
							case TALK:
							case AUTO:
								if (MsgClass == RAWKEY) ToggleGadget(GadgetNumber);
								i = GetGadgetState(GadgetNumber);
								SwitchNumber(GadgetNumber, i);
								break;

							case DF0S:
							case DF1S:
							case DF2S:
							case DF3S:
								if (MsgClass == RAWKEY) ToggleGadget(GadgetNumber);

								if (GetGadgetState(GadgetNumber)) {
									if (Source == GadgetNumber) break;
									if (Source>=0) SelectGadget(DF0S+Source, FALSE);
									if (GrabUnits(PresentDisks)) Source = GadgetNumber;
									else DeselectDiskGadgets();
								}
								else if (GadgetNumber == Source) Source = -1;
								if (!Dest && Source<0) DropUnits();
								ResetAuto();
								break;

							case DF0D:
							case DF1D:
							case DF2D:
							case DF3D:
								i = GadgetNumber-DF0D;
								if (MsgClass == RAWKEY) ToggleGadget(GadgetNumber);

								if (GetGadgetState(GadgetNumber)) {
									if (GrabUnits(PresentDisks)) Dest |= MASK(i);
									else DeselectDiskGadgets();
								}
								else Dest &= ~MASK(i);
								if (!Dest && Source<0) DropUnits();
								ResetAuto();
								break;

							case RETRY:
							case VDNUMBER:
							case XPKLIB:
							case LABEL:
								if (MsgClass == RAWKEY) ActivateStringGadget(GadgetNumber);
								break;

							default:
								break;
						}
						break;
				}
			}

			ASSERT(!RexxMsg);

			if (ARexxPort && (RemRexx = RexxMsg = (void *)GetMsg(ARexxPort))) {
				if (RexxMsg->rm_Node.mn_Node.ln_Type == NT_REPLYMSG) {
					ASSERT(PendingMacros > 0);
					AfterARexxMacro(RexxMsg);
					RexxMsg = NULL;
					--PendingMacros;
				}
				else {
					unsigned long *Par;

					if ((Op = ParseARexx(RexxMsg, &Par)) > 0) {

						if (Op<=COPYMODE && !GadgetEnabled(Op)) SetRexxError(RexxMsg, 1);
						else {
							p = Par ? (char *)Par[0] : NULL;

							switch(Op) {

								case COPYMODE:
									ASSERT(Par);
									for(i=0; i<BUFFER_COUNT; i++) {
										if (Par[i]) {
											GetBuffer(i);
											SetCopyMode(Buffer);
											EnableGadgets(CurrentMask = GadgetMask & TypeMask[Buffer]);
											if (Buffer != i) SetRexxError(RexxMsg, 5);
											break;
										}
									}
									break;

								case CHECK:
								case READ:
								case FORMAT:
								case COPY:
								case WRITE:
									if (!(i = SwitchActions(Op))) PendingRexxMsg = RexxMsg;
									else SetRexxError(RexxMsg, i);
									break;

								case QUIT:
									goto GameOver;
									break;

								case STOP:
									Stop();
									break;
								case NOWB:
									if (Par[ON]) j = 1;
									else if (Par[OFF]) j = 0;

									if (KillSys(j)) {
										SetRexxError(RexxMsg,	20);
										goto GameOver;
									}
									break;

								case ICONIFY:
									Iconify(Par[ON] ? TRUE : FALSE);
									break;

								case SAVECONF:
									SaveConfig();
									break;

								case VERIFY:
								case COMPRESSION:
								case DATE:
								case INCNAME:
								case FFS:
								case INTL:
								case DIRCACHE:
								case TALK:
								case AUTO:
								case REQUESTERS:
								case PRINTERRORS:
								case SMARTREFRESH:
								case RECALIBRATECHECK:
									ASSERT(Par);
									if (Par[ON]) i = 1;
									else if (Par[OFF]) i = 0;

									switch(Op) {
										case REQUESTERS:
											Requesters = i;
											break;
										case RECALIBRATECHECK:
											RecalibrateCheck = i;
											break;
										case SMARTREFRESH:
											SmartRefresh = i;
											break;
										case ICONIFY:
											Iconify(i);
											break;
										default:
											SelectGadget(Op, i);
											SwitchNumber(Op, i);
											break;
									}
									break;

								case LABEL:
									ASSERT(Par);
									if (SetDiskName(p)) SetRexxError(RexxMsg, 1);
									break;

								case XPKLIB:
									ASSERT(Par);
									if (SetXPKLibName(p)) SetRexxError(RexxMsg, 1);
									break;

								case VDNUMBER:
									ASSERT(Par);
									if (SetVDNumber((USHORT)*((ULONG *)(Par[0])))) SetRexxError(RexxMsg, 1);
									break;

								case VDNAME:
									ASSERT(Par);
									if (SetDeviceName(p)) SetRexxError(RexxMsg, 1);
									break;

								case FILENAME:
									ASSERT(Par);
									if (SetBufferName(p)) SetRexxError(RexxMsg, 1);
									break;

								case RETRY:
									ASSERT(Par);
									if (SetRetryNumber((USHORT)*((ULONG *)(Par[0])))) SetRexxError(RexxMsg, 1);
									break;

								case SCYL:
									ASSERT(Par);
									if (SetSCyl((USHORT)*((ULONG *)(Par[0])))) SetRexxError(RexxMsg, 1);
									break;

								case ECYL:
									ASSERT(Par);
									if (SetECyl((USHORT)*((ULONG *)(Par[0])))) SetRexxError(RexxMsg, 1);
									break;

								case SOURCE:
									ASSERT(Par);
									if (Par[DRIVE] && Par[SOURCEOFF]) {
										SetRexxError(RexxMsg, 10);
										break;
									}
									if (Par[SOURCEOFF]) {
										if (Source>=0) SelectGadget(DF0S+Source, FALSE);
										Source = -1;
									}
									else {
										Op = *((unsigned long *)(Par[DRIVE]));
										if (Op>3 || Op<0) {
											SetRexxError(RexxMsg,10);
											break;
										}
										if (MASK(Op) & ~PresentDisks) {
											SetRexxError(RexxMsg,2);
											break;
										}
										if (Source == Op) break;
										if (Source>=0) SelectGadget(DF0S+Source, FALSE);
										if (GrabUnits(PresentDisks)) {
											Source = Op;
											SelectGadget(DF0S+Op, TRUE);
										}
										else {
											Source = -1;
											SetRexxError(RexxMsg, 5);
										}
									}
									if (!Dest && Source<0) DropUnits();
									ResetAuto();
									break;
								case DESTINATION:
									ASSERT(Par);
									if (Par[DESTON] && Par[DESTOFF]) {
										SetRexxError(RexxMsg, 10);
										break;
									}

									if (Par[DRIVES]) {
										if (!(Par[DESTON] || Par[DESTOFF])) Dest = 0;

										i = 0;
										while (((unsigned long **)Par[DRIVES])[i]) {
											Op = *(((unsigned long **)Par[DRIVES])[i++]);

											if (Op > 3 || Op < 0) {
												SetRexxError(RexxMsg, 10);
												break;
											}
											if ((1<<Op) & ~PresentDisks) {
												SetRexxError(RexxMsg, 2);
												break;
											}

											if (Par[DESTOFF]) Dest &= ~(1<<Op);
											else Dest |= 1<<Op;
										}
									}
									else {
										if (Par[DESTON]) Dest = PresentDisks;
										else Dest = 0;
									}

									for(i=0; i<4; i++) SelectGadget(DF0D+i, Dest & (1<<i));
									if (Dest && !GrabUnits(PresentDisks)) {
										DeselectDiskGadgets();
										Dest = 0;
										SetRexxError(RexxMsg, 5);
									}
									if (!Dest && Source<0) DropUnits();
									ResetAuto();
									break;
								case AREXX:
									ASSERT(Par);
									if (StartARexxMacro(CreateArgstring(p, strlen(p)+1), FALSE))
										PendingMacros++;
									else SetRexxError(RexxMsg, 1);
									break;
								case HELP:
									PrintARexxHelp(RexxMsg, p);
									break;
								case WINDOW:
									ASSERT(Par);
									if (!Par[NAMES]) {
										SetRexxError(RexxMsg, 10);
										break;
									}
									{
										char **Names = (char **)Par[NAMES];
										j = -1;

										while (Names[++j]) {
											if (!Stricmp(Names[j], WindowName[WIN_MAIN])) {
												if (Par[OPENIT]) SetUpWindow();
												i = WIN_MAIN;
											}
											else if (!Stricmp(Names[j], WindowName[WIN_INFO])) {
												if (Par[OPENIT]) SetUpInfoWindow();
												i = WIN_INFO;
											}
											else if (!Stricmp(Names[j], WindowName[WIN_OPTIONS])) {
												if (Par[OPENIT]) SetUpOptWindow();
												i = WIN_OPTIONS;
											}
											else {
												SetRexxError(RexxMsg, 10);
												break;
											}

											if (Par[LEFTEDGE] || Par[TOPEDGE]) ChangeWindowSize(i, Par[LEFTEDGE] ? (int)*((ULONG *)Par[LEFTEDGE]) : -1, Par[TOPEDGE] ? (int)*((ULONG *)Par[TOPEDGE]) : -1);

											if (Par[MINIT]) ZoomWindow(i, 1);
											if (Par[MAXIT]) ZoomWindow(i, 0);

											if (!WindowPointer(i)) {
												if (Par[ACTIVATEIT] || Par[FRONT] || Par[BACK] || Par[CLOSEIT])
													SetRexxError(RexxMsg, 1);
												break;
											}

											if (Par[ACTIVATEIT]) ActivateWindow(WindowPointer(i));
											if (Par[FRONT]) WindowToFront(WindowPointer(i));
											if (Par[BACK]) WindowToBack(WindowPointer(i));
											if (Par[CLOSEIT]) {
												if (i == WIN_MAIN) CloseAllWindows();
												else CloseWin(i);
											}
										}
									}
									break;
							}
						}
					}
					if (PendingRexxMsg != RexxMsg) ReplyMsg((void *)RexxMsg);
					RexxMsg = NULL;
				}
			}

			if (AppMsg = GetMsg(AppPort)) {
				ReplyMsg(AppMsg);
				Iconify(FALSE);
			}

			ASSERT(!RexxMsg);
			ASSERT(WindowPort || ARexxPort);

		} while(AppMsg || (WindowPort && Msg) || (ARexxPort && RemRexx) || ((Dest || Source>=0) && !Copying && CheckClick()) || AutoFlag);

		Signals = Wait(CopierMask | (WindowPort ? MASK(WindowPort->mp_SigBit) : 0) | MASK(AppPort->mp_SigBit) | MASK(ClickPort->mp_SigBit) | (ARexxPort ? MASK(ARexxPort->mp_SigBit) : 0));
	}

GameOver:

	Done = TRUE;
	if (DiskObj) FreeDiskObject(DiskObj);
	if (RA) FreeArgs(RA);
	if (Copying) {
		Signal(Copier, StopMask);
		Wait(CopierMask);
	}
	if (Copier) {
		Signal(Copier, StartMask);
		Wait(CopierMask);
	}
	if (PendingRexxMsg) {
		SetRexxError(PendingRexxMsg, 30);
		ReplyMsg(PendingRexxMsg);
	}
	while (PendingMacros) {
		WaitPort(ARexxPort);
		while((RemRexx = (void *)GetMsg(ARexxPort)) && RemRexx->rm_Node.mn_Node.ln_Type != NT_REPLYMSG) {
			SetRexxError(RemRexx, 30);
			ReplyMsg(RemRexx);
		}
		if (RemRexx) {
			AfterARexxMacro(RemRexx);
			PendingMacros--;
		}
	}
	FreeBuffers[Buffer]();
	FreeCompBuffer();
	CloseHyperHelp();
	RemovePostedDRU();
	DropUnits();
	SetTaskPri(Me, OldPri);
	RemoveClick();
	CloseAllWindows();
	CloseIcon();
	ShutScreen();
	FreeDiskList();
	CloseErrorConsole();
	CloseVoice();
	CloseTimer();
	if (RexxMsg) ReplyMsg(RexxMsg);
	FreeStuff();
	CloseARexxConsole();
	CloseLibrary(IconBase);
	return(0);
}

