#include "SD.h"
#include <libraries/xpk.h>
#include <libraries/iffparse.h>

#define STACKSIZE 4096

static void Dummy(void);
static unsigned char ReAllocRAMBuffers(void);
static unsigned char ReAllocHDBuffer(void);
static unsigned char ReAllocVDisk(void);
static void FreeRAMBuffers(void);

unsigned char (*ReAllocBuffers[4])(void) = { (unsigned char (*)(void))Dummy, ReAllocRAMBuffers, ReAllocHDBuffer, (unsigned char (*)(void))Dummy };
void (*FreeBuffers[4])(void) = { Dummy, FreeRAMBuffers, Dummy, Dummy };

ULONG	StoreMask;		/* What to signal to Copier when the store/retrieve is finished */

static ULONG StoreCoMask;	/* What to signal back to the copy task */

unsigned long DMACheck[8];		/* Checksums for the DMA buffer */

static unsigned char *RAMBuffer[80];	/* pointers to the 80 buffer chunks */
static unsigned long Check[160];			/* Checksums of the internal RAM buffer */

static const	ULONG	SYNC = 0x4489;
static const	ULONG	SYNCSYNC = 0x44894489;
static const	ULONG	MFMZeroes = 0xAAAAAAAA;
static const	ULONG	MFMMask = 0x55555555;

#define ID_XPKF MAKE_ID('X','P','K','F')
#define ID_SDDD MAKE_ID('S','D','D','D')
#define ID_SDHD MAKE_ID('S','D','H','D')
#define ID_BODY MAKE_ID('B','O','D','Y')

static ULONG FORM_DD[3] = { ID_FORM, 0, ID_SDDD };
static ULONG FORM_HD[3] = { ID_FORM, 0, ID_SDHD };
static ULONG BODY[2] = { ID_BODY, 0 };

static unsigned char TaskName[] = "SD_Store",
							Name[31], 								/* The disk name */
							VDiskOpen,								/* Is a virtual disk open? */
							Storing, 								/* Are we Storing? */
							TaskAlive, 								/* Is the compression task alive? */
							StopStore,								/* When TRUE, we have to abort */
							NotCompressed[10],					/* Which cylinders haven't been compressed? */
							*InBuf, *OutBuf,						/* Various pointers for xpk */
							*BlockPointers[2][22*2];			/* Pointers to the disk block data for
																			this store pass */
static unsigned char ReadID[] = "Read",
							VerifyID[] = "Verify",
							UnitTrackMsg[] = "%s error on unit %ld, track %ld: %s\n",
							NoFirstSector[] = "Can't find first sector after gap",
							WrongSecNumber[] = "Wrong sector number",
							WrongTrackNumber[] = "Wrong track number",
							WrongNextNumber[] = "Wrong number of sectors before gap",
							BadChecksum[] = "Bad checksum",
							VerifyError[] = "Failed verify",
							ErrorName[] = "<BAD NAME>";

static unsigned short	CompCyl;			/* The cylinder we are Storing */

static struct IOStdReq *VIOStdReq;		/* The IOReq for the virtual disk */
static struct MsgPort *VPort;
static ULONG DosMark;						/* The DOS mark of the current disk */
static struct Process *Process;

static unsigned char ConsoleName[] = "CON:0/130/600/70/SuperDuper Error Report/AUTO/CLOSE";
static long ErrorConsole;

static void PrintError(unsigned char *Operation, unsigned int Unit, unsigned int Track, unsigned char *ErrorMsg) {

	if (!(PrintErrors && (stdout || ErrorConsole))) return;
	ObtainSemaphore(&Semaphore[WIN_OPTIONS]);
	sprintf(s, UnitTrackMsg, Operation, (ULONG)Unit, (ULONG)Track, ErrorMsg);
	Write(stdout ? stdout : ErrorConsole, s, strlen(s));
	ReleaseSemaphore(&Semaphore[WIN_OPTIONS]);
}

void OpenErrorConsole(void) {
	if (!ErrorConsole && !stdout) ErrorConsole = Open(ConsoleName, MODE_NEWFILE);
}

void CloseErrorConsole(void) {
	if (ErrorConsole) Close(ErrorConsole);
	ErrorConsole = NULL;
}


static void CloseVDisk(void) {

	if (VDiskOpen) {
		CloseDevice(VIOStdReq);
		VDiskOpen = 0;
	}
	DeleteIORequest(VIOStdReq);
	VIOStdReq = NULL;
	DeleteMsgPort(VPort);
	VPort = NULL;
}

static unsigned char OpenVDisk(void) {

	if ((VPort = CreateMsgPort()) &&
		(VIOStdReq = CreateIORequest(VPort, sizeof(struct IOStdReq))) &&
		(VDiskOpen = !OpenDevice(GetDeviceName(), GetVDNumber(), VIOStdReq, 0))) return(1);
	else {
		CloseVDisk();
		return(0);
	}
}

#define ERROR(x) { StoreError = ERR_##x; goto ErrorExit; }

/* This is the task running while we compress */

static void __saveds Compress(void) {

	short i;
	unsigned long CompLen;
	signed char CompCoStore;
	char *Temp = NULL;
	BPTR h = 0;
	ULONG Size = 4, Pos = 0;

	StoreCoMask = MASK(CompCoStore = AllocSignal(-1));

	StopStore = 0;
	setmem(NotCompressed, sizeof(NotCompressed), 0);

	switch(Buffer) {

		case RAM_BUFFER:
			FreeRAMBuffers();
			break;

		case HD_BUFFER:
			if (!(h = Open(GetBufferName(), MODE_READWRITE))) ERROR(CANTOPENFILE);
			if (!Compression && !(Temp = AllocVec(RAMBufferBlock, MEMF_PUBLIC))) ERROR(IO);

			ASSERT(h)
			ASSERT(Compression || Temp)

			ASSERT(CurrentType == DRT_AMIGA || CurrentType == DRT_150RPM)

			if (Write(h, CurrentType == DRT_AMIGA ? FORM_DD : FORM_HD, 12)<12) ERROR(IO);
			break;

		case VDISK_BUFFER:
			if (!OpenVDisk()) ERROR(CANTOPENVDISK);
			break;
	}

	do {
		if (Buffer == RAM_BUFFER) {
			Check[CompCyl*2] = CheckBlocks(BlockPointers[CompCyl%2], SPT);
			Check[CompCyl*2+1] = CheckBlocks(&BlockPointers[CompCyl%2][SPT], SPT);
		}

		if (Compression && Buffer != VDISK_BUFFER) {

			void *FakeBuffer = AllocVec(RAMBufferBlock, MEMF_PUBLIC);

			ASSERT(CompCyl < 80)
			ASSERT(InBuf && OutBuf)

			for(i=0; i<SPT*2; i++)
				CopyMem(BlockPointers[CompCyl%2][i], InBuf+i*TD_SECTOR, TD_SECTOR);

			i  = XpkPackTags(XPK_InBuf, InBuf,
								XPK_OutBuf, OutBuf,
								XPK_InLen, (ULONG)(RAMBufferBlock),
								XPK_OutBufLen, (ULONG)(RAMBufferBlock+RAMBufferBlock/32+2*XPK_MARGIN),
								XPK_PackMethod, GetXPKLibName(),
								XPK_GetOutLen, &CompLen,
								TAG_DONE);

			FreeVec(FakeBuffer);

			if (i == XPKERR_NOMEM) StopStore = TRUE;
			else if (!i || i == XPKERR_EXPANSION) {
				if (i == XPKERR_EXPANSION || CompLen>RAMBufferBlock) {
					NotCompressed[CompCyl/8] |= (1<<(CompCyl%8));
					if (Buffer == HD_BUFFER) {
						BODY[1] = RAMBufferBlock;
						if (Write(h, BODY, 8)<8 || Write(h, InBuf, RAMBufferBlock)<RAMBufferBlock)
							StopStore = TRUE;
						else Size += 8+RAMBufferBlock;
					}
					else {
						if (RAMBuffer[CompCyl] = AllocVec(RAMBufferBlock, MEMF_PUBLIC)) CopyMemQuick(InBuf, RAMBuffer[CompCyl], RAMBufferBlock);
						else StopStore = TRUE;
					}
				}
				else {
					if (Buffer == HD_BUFFER) {
						if (Write(h, OutBuf, CompLen)<CompLen)
							StopStore = TRUE;
						else Size += CompLen;
					}
					else {
						if (RAMBuffer[CompCyl] = AllocVec(CompLen, MEMF_PUBLIC)) CopyMemQuick(OutBuf, RAMBuffer[CompCyl], CompLen);
						else StopStore = TRUE;
					}
				}
			}
			else {
				ERROR(COMP);
			}
		}
		else {
			switch(Buffer) {

				case RAM_BUFFER:
					if (RAMBuffer[CompCyl] = AllocVec(RAMBufferBlock, MEMF_PUBLIC)) {
						for(i=0; i<SPT*2; i++)
							CopyMem(BlockPointers[CompCyl%2][i], RAMBuffer[CompCyl]+i*TD_SECTOR, TD_SECTOR);
					}
					else StopStore = TRUE;
					break;

				case HD_BUFFER:
					BODY[1] = RAMBufferBlock;
					for(i=0; i<SPT*2; i++)
						CopyMem(BlockPointers[CompCyl%2][i], Temp+i*TD_SECTOR, TD_SECTOR);
					if (Write(h, BODY, 8)<8 || Write(h, Temp, RAMBufferBlock)<RAMBufferBlock) StopStore = TRUE;
					else Size += 8+RAMBufferBlock;
					break;

				case VDISK_BUFFER:
					for(i=0; i<SPT*2; i++) {
						VIOStdReq->io_Command = CMD_WRITE;
						VIOStdReq->io_Length = TD_SECTOR;
						VIOStdReq->io_Offset = Pos*2*(ULONG)TrackSize+i*(ULONG)TD_SECTOR;
						VIOStdReq->io_Data = BlockPointers[CompCyl%2][i];
						if (DoIO(VIOStdReq)) StopStore = TRUE;
					}
					Pos++;
					break;

			}
		}

		Signal(Copier, StoreMask);
		Wait(StoreCoMask);
	} while(!StopStore);

	if (Buffer == HD_BUFFER) {
		SetFileSize(h, Size+8, OFFSET_BEGINNING);
		Seek(h, 4, OFFSET_BEGINNING);
		Write(h, &Size, 4);
		Close(h);
		if (Temp) FreeVec(Temp);
	}
	else if (Buffer == VDISK_BUFFER) CloseVDisk();

	FreeSignal(CompCoStore);
	return;

ErrorExit:
	StopStore = TRUE;
	if (h) Close(h);
	if (Temp) FreeVec(Temp);
	CloseVDisk();
	Signal(Copier, StoreMask);
	Wait(StoreCoMask);
	FreeSignal(CompCoStore);
}

/* This is the task running while we decompress */

static void __saveds Decompress(void) {

	long i;
	signed char CompCoStore;
	char *DataBuf = NULL;
	ULONG Temp[3], Pos = 0;
	BPTR h = NULL;

	StoreCoMask = MASK(CompCoStore = AllocSignal(-1));

	StopStore = 0;

	if (Buffer == HD_BUFFER) {
		ASSERT(CurrentType == DRT_AMIGA || CurrentType == DRT_150RPM)

		if (!(h = Open(GetBufferName(), MODE_OLDFILE))) ERROR(CANTOPENFILE);
		if (Read(h, Temp, 12)<12) ERROR(IO);
		if (Temp[0] != ID_FORM) ERROR(IFF);
		if (!((CurrentType == DRT_AMIGA && Temp[2] == ID_SDDD) || (CurrentType == DRT_150RPM && Temp[2] == ID_SDHD)))
			ERROR(WRONGTYPE);
		setmem(NotCompressed, sizeof(NotCompressed), 0);
	}
	else if (Buffer == VDISK_BUFFER) {
		if (!OpenVDisk()) ERROR(CANTOPENVDISK);
	}


	do {
		ASSERT(CompCyl < 80)
		ASSERT(Buffer != RAM_BUFFER || RAMBuffer[CompCyl])
		ASSERT(Buffer != HD_BUFFER || h)

		if (Buffer == HD_BUFFER) {
			do {
				if ((i = Read(h, Temp, 8))<8) ERROR(IO);
				if (Temp[0] == ID_BODY) {
					NotCompressed[CompCyl/8] |= (1<<(CompCyl%8));
					if (Compression && Read(h, DataBuf = OutBuf, RAMBufferBlock)<RAMBufferBlock) ERROR(IO);
				}
				else if (Temp[0] == ID_XPKF) {
					if (Compression) {
						memcpy(InBuf, Temp, 8);
						if (Read(h, InBuf+8, Temp[1])<Temp[1]) ERROR(IO)
					}
					else ERROR(NOCOMP);
				}
				else {
					if (Seek(h, Temp[1], OFFSET_CURRENT)<0) ERROR(IO)
				}
			} while(Temp[0] != ID_BODY && Temp[0] != ID_XPKF);
		}

		if (!Compression || (NotCompressed[CompCyl/8] & (1<<(CompCyl%8)))) {
			if (Buffer == RAM_BUFFER) DataBuf = RAMBuffer[CompCyl];
		}
		else if (Buffer != VDISK_BUFFER) {

			ASSERT(InBuf && OutBuf);
			ASSERT(Buffer != HD_BUFFER || *((ULONG *)InBuf) == ID_XPKF)
			ASSERT(Buffer != RAM_BUFFER || *((ULONG *)(RAMBuffer[CompCyl])) == ID_XPKF)

			i = XpkUnpackTags(XPK_InBuf, Buffer == HD_BUFFER ? InBuf : RAMBuffer[CompCyl],
								XPK_InLen, (ULONG)(RAMBufferBlock),
								XPK_OutBuf, OutBuf,
								XPK_OutBufLen, (ULONG)(RAMBufferBlock+RAMBufferBlock/32+2*XPK_MARGIN),
								TAG_DONE);
			DataBuf = OutBuf;
			if (i == XPKERR_CHECKSUM || i == XPKERR_CORRUPTPKD) ERROR(CHECKSUM);
			if (i) ERROR(COMP);
		}

		for(i=0; i<SPT*2; i++) {
			if (i == SPT) {
				if (Buffer == RAM_BUFFER && Check[CompCyl*2] != CheckBlocks(BlockPointers[CompCyl%2], SPT))
					ERROR(CHECKSUM);
				Signal(Copier, StoreMask);
				Wait(StoreCoMask);
				if (StopStore) goto FreeAndExit;
			}
			if (DataBuf)
				CopyMemQuick(DataBuf+i*TD_SECTOR, BlockPointers[CompCyl%2][i], TD_SECTOR);
			else if (Buffer == HD_BUFFER) {
				if (Read(h, BlockPointers[CompCyl%2][i], TD_SECTOR)<TD_SECTOR) ERROR(IO);
			}
			else if (Buffer == VDISK_BUFFER) {
				VIOStdReq->io_Command = CMD_READ;
				VIOStdReq->io_Length = TD_SECTOR;
				VIOStdReq->io_Offset = Pos*2*(ULONG)TrackSize+i*(ULONG)TD_SECTOR;
				VIOStdReq->io_Data = BlockPointers[CompCyl%2][i];
				if (DoIO(VIOStdReq)) ERROR(IO);
			}
		}

		Pos++;

		if (Buffer == RAM_BUFFER && Check[CompCyl*2+1] != CheckBlocks(&BlockPointers[CompCyl%2][SPT], SPT))
			ERROR(CHECKSUM);

		Signal(Copier, StoreMask);
		Wait(StoreCoMask);
	} while(!StopStore);

FreeAndExit:
	if (Buffer == HD_BUFFER) Close(h);
	else if (Buffer == VDISK_BUFFER) CloseVDisk();

	FreeSignal(CompCoStore);
	return;

ErrorExit:
	StopStore = TRUE;
	if (h) Close(h);
	CloseVDisk();
	Signal(Copier, StoreMask);
	Wait(StoreCoMask);
	FreeSignal(CompCoStore);
}


unsigned short ReAllocCompBuffer(void) {
	static ULONG CurrentCompType = DRT_EMPTY;

	if (InBuf && CurrentCompType == CurrentType) return(TRUE);

	ASSERT(CurrentType == DRT_AMIGA || CurrentType == DRT_150RPM)

	FreeCompBuffer();

	D(printf("Reallocating compression buffer to type %lx\n", CurrentType);)

	if (InBuf = AllocVec(RAMBufferBlock*2+RAMBufferBlock/32+2*XPK_MARGIN, MEMF_PUBLIC)) {
		OutBuf = InBuf+RAMBufferBlock;
		CurrentCompType = CurrentType;
		return(TRUE);
	}
	else {
		CurrentCompType = DRT_EMPTY;
		return(FALSE);
	}
}

void FreeCompBuffer(void) {
	FreeVec(InBuf);
	InBuf = OutBuf = NULL;
}



/* Here we start the comp/decomp task. If it's still alive, we simply
	signal it. All the following functions up to the Alloc/Free series
	can only be called from the copy task. */

static void StartTask(void Code()) {

 	if (TaskAlive) {
		ASSERT(Process)
		Signal(Process, StoreCoMask);
	}
	else {
		Process = CreateNewProcTags(NP_Entry, Code, NP_Name, TaskName,
											NP_Priority, Compression && Buffer != VDISK_BUFFER ? 0L : 1L, TAG_DONE);
		TaskAlive = 1;
	}
	Storing = 1;
}


/* We start the comp/decomp */

void StartStore(unsigned short Track) {
	ASSERT(Track<160)
	CompCyl = Track/2;
	StartTask(Compress);
}


void StartRetrieve(unsigned short Track, unsigned char *B1, unsigned char *B2) {
	short i;
	ASSERT(Track<160)
	CompCyl = Track/2;
	if (B1 && B2) {
		for(i=0; i<SPT; i++) BlockPointers[CompCyl%2][i] = B1+SE_Data+TD_SECTOR+i*BLOCKSIZE;
		for(i=0; i<SPT; i++) BlockPointers[CompCyl%2][i+SPT] = B2+SE_Data+TD_SECTOR+i*BLOCKSIZE;
	}
	StartTask(Decompress);
}

/* We wait for the end of compression. We get a non-zero result if compression
	is over for lack of space. */

unsigned char WaitStore(unsigned char Rem) {

	if (Storing) {
		ASSERT(TaskAlive)
		Wait(StoreMask);
		Storing = 0;
		if (Rem || StopStore) {
			Rem = StopStore;
			TaskAlive = 0;
			StopStore = TRUE;
			ASSERT(Process)
			Signal(Process, StoreCoMask);
			return(Rem);
		}
	}
	return(0);
}


/* All the following allocation/deallocation functions should be called from the main task. */

static void Dummy(void) {}

/*	This routine frees RAM buffers and compression tables. */

static void FreeRAMBuffers(void) {

	register short i;

	for(i=0; i<80; i++) {
		FreeVec(RAMBuffer[i]);
		RAMBuffer[i] = NULL;
	}
}

/* Here we allocate buffers and, if we don't get enough, we allocate compression stuff. */

static unsigned char ReAllocRAMBuffers(void) {

	FreeRAMBuffers();
	FlushMem();
	return(TRUE);
}


static unsigned char ReAllocHDBuffer(void) {
	return(TRUE);
}


/* Here we open any exec device. */

static unsigned char ReAllocVDisk(void) {

	unsigned char result = 0;
	struct IOStdReq *IOStdReq;

	if (!GetDeviceName()) return(0);
	if (IOStdReq = CreateStdIO(TPort)) {
		if (!OpenDevice(GetDeviceName(), GetVDNumber(), IOStdReq, 0)) {
			result = 1;
			CloseDevice(IOStdReq);
		}
		DeleteStdIO(IOStdReq);
	}
	return(result);
}


/* The following functions are just computations, and can be called
	from everywhere. From StoreTrack afterwards, the blitter is used,
	so you have to call from the copy task. */


/*	This routine recalculates the checksum for a block, and updates it in
	word 5.  The sum of all the words in a block must be 0. */

static void ReCheck(long *w) {

	register short i;
	register long CheckSum;

	CheckSum = 0;
	for (i=0; i<128; i++)
		CheckSum += w[i];
	w[5] -= CheckSum;
}

/* We simply DateStamp the creation date, modification date, and
	rechecksum the block. A little random factor is added since we
	update many disks in a short period of time. We also increment the name of
	the disk if required. */

static void UpdateRootBlock(unsigned char *b, unsigned char Date, unsigned char IncName) {

	static unsigned char ShiftFactor = 1;

	if (Date){
		ShiftFactor = (ShiftFactor+2)%50;
		DateStamp((void *)(b + 420));
		DateStamp((void *)(b + 484));
		((struct DateStamp *)(b + 420))->ds_Tick = (((struct DateStamp *)(b + 420))->ds_Tick+ShiftFactor)%3000;
		((struct DateStamp *)(b + 484))->ds_Tick = (((struct DateStamp *)(b + 484))->ds_Tick+ShiftFactor)%3000;
	}
	if (IncName) {
		IncrementDiskName();
		strcpy((char *)(b+433), GetDiskName());
		*(b+432) = strlen(b+433);
	}
	ReCheck((void *)b);
}

/*
 *	  This routine creates data for the format option.	 Note the clever
 *	  way the data is built up; this routine should build a disk exactly
 *	  the same way the standard AmigaDOS format does.	If we are on track
 *	  40, some additional work must be done to create a root block and
 *	  bitmap, but it's not too bad. I don't know if this is the right
 *   way to do under FFS, but since it works . . .
 */
static void MakeFormatData(unsigned short Sector, unsigned char *b, unsigned char UseFFS, unsigned char IntlFS, unsigned char DirCache) {

	register unsigned long *p, i;
	register unsigned char *q;

	p = (unsigned long *)b;
	memset(b, 0, TD_SECTOR);

	ASSERT(b)

	if (Sector == 0)
		p[0] =	(ULONG)'D'<<24|(ULONG)'O'<<16|(ULONG)'S'<<8|
					(ULONG)((DirCache ? 4 : (IntlFS ? 2  : 0 )) + (UseFFS ? 1 : 0));
	else if (Sector == RootBlock) {
		p[0] = 2;
		p[3] = 0x48;
		p[78] = 0xffffffff;
		p[79] = RootBlock+1+(DirCache != 0);
		q = (unsigned char *)(p + 108);
		strcpy((char *)(q+1), GetDiskName());
		*q = strlen(q+1);
		p[127] = 1;
		if (DirCache) p[126] = RootBlock+1;
		UpdateRootBlock(b, TRUE, FALSE);
	}
	else if (!DirCache && Sector == RootBlock+1 || DirCache && Sector == RootBlock+2) {
		if (CurrentType == DRT_AMIGA) {
			for (i=1; i<55; i++)
				p[i] = 0xffffffffUL;
			p[0] = DirCache ? 0xc001c037UL : 0xc000c037UL;
			p[28] = DirCache ? 0xfffe3fffUL : 0xffff3fffUL;
			p[55] = 0x3fffffffUL;
		}
		else {
			for (i=1; i<110; i++)
				p[i] = 0xffffffffUL;
			p[0] = DirCache ? 0x8000006fUL : 0x8000006eUL;
			p[55] = 0x3fffffffUL;
			p[110] = 0x3fffffffUL;
			if (DirCache) p[56] = 0xfffffffe;
		}
	}
	else if (DirCache && Sector == RootBlock+1) {
		p[0] = 0x21;
		if (CurrentType == DRT_AMIGA) {
			p[1] = 0x371;
			p[2] = 0x370;
			p[5] = 0xfffff8feUL;
		}
		else {
			p[1] = 0x6e1;
			p[2] = 0x6e0;
			p[5] = 0xfffff21eUL;
		}
	}
}


/* Various obvious service functions */

static void SetBufferStart(unsigned char *Buf) {
	*((ULONG *)(Buf)) = MFMZeroes;
	*((ULONG *)(Buf+SE_Sync)) = SYNCSYNC;
}


static void SetBufferBorders(unsigned char *Buf) {
	*((ULONG *)(Buf-4)) = *((ULONG *)(Buf-8)) = *((ULONG *)(Buf+SPT*BLOCKSIZE)) = MFMZeroes;
}

signed char FindOffset(unsigned char *Buf) {
	if (*((ULONG *)(Buf)) == SYNCSYNC) return(-4);
	else if (*((USHORT *)(Buf)) == SYNC) return(-6);
	else return(-8);
}

/* Calculates the gap length of the given Buf (not offset). If anything
	is wrong, GapSize is returned. */

unsigned short GapLength(unsigned char *Buf) {

	unsigned char NumSec;
	unsigned char *p;

	NumSec = (unsigned char)(DecodeLong((Buf+=FindOffset(Buf))+SE_Header));
	if (NumSec<1 || NumSec>SPT) return(GapSize);

	Buf += NumSec*BLOCKSIZE;
	if (p = FindSync(Buf, GapSize)) return((unsigned short)(max(p-Buf-4,MinGap)+GapAdd));
	else return(GapSize);
}


/* We decode a track and we store it. If ReallyStore is zero we simply check it. From
	here, we're using the blitter, so you have to call from the copy task.
	We return 0 if everything is OK, 1 on error. */

signed char StoreTrack(unsigned char *p, unsigned char Unit, unsigned short Track) {

	register unsigned char i;
	unsigned char Sec, NumSec;
	signed char Result = 0;

	if (Track>159) return(1);
	if (Track == 0) DosMark = 0;

	p += FindOffset(p);
	NumSec = (unsigned char)(DecodeLong(p+SE_Header));
	if ((unsigned char)(DecodeLong(p+SE_Header)>>16) != Track) {
		PrintError(ReadID, Unit, Track, WrongTrackNumber);
		Result = 1;
	}

	if (NumSec>0 && NumSec<=SPT)
		for(i=0; i<SPT; i++) {
			if (i == NumSec) {
				if ((p = FindSync(p, GapSize)) == NULL) {
					PrintError(ReadID, Unit, Track, NoFirstSector);
					Result = 1;
					break;
				}
				p += FindOffset(p);
			}
			Sec = (unsigned char)(DecodeLong(p+SE_Header)>>8);
			if (Sec>=SPT) {
				PrintError(ReadID, Unit, Track, WrongSecNumber);
				Result = 1;
			}
			else {
				if (CheckCheckSums(p, 1)) {
					PrintError(ReadID, Unit, Track, BadChecksum);
					Result = 1;
				}
				DecodeSector(p+SE_Data, &EBN);

				BlockPointers[(Track/2)%2][(Track%2)*SPT+Sec] = p+SE_Data+TD_SECTOR;

				if (Track == 0 && Sec == 0) DosMark = *((ULONG *)(p+SE_Data+TD_SECTOR));
				if (Track == 80 && Sec == 0) {
					for(Sec=0; Sec<30; Sec++) Name[Sec] = p[SE_Data+TD_SECTOR+433+Sec];
					if ((Sec = (unsigned char)(p[SE_Data+TD_SECTOR+432]))>30) strcpy(Name, ErrorName);
					else Name[Sec] = 0;
				}
			}
			p += BLOCKSIZE;
		}
	else {
		PrintError(ReadID, Unit, Track, WrongNextNumber);
		Result = 1;
	}

	return(Result);
}


/* We take a track from the RAM buffer and we create a MFM one */

signed char RetrieveTrack(unsigned short Track, unsigned char *Buf) {

	register unsigned char i;
	register unsigned char *p;

	if (Track>159) return(1);

	for(i=0; i<SPT; i++) {
		p = Buf+i*BLOCKSIZE;
		SetBufferStart(p);
		EncodeLong(0xFF000000UL | (ULONG)Track<<16 | i<<8 | SPT-i, p+SE_Header);
		setmem(p+SE_Label, TD_LABELSIZE*2, 0xAA);
		if (Track == 0 && i == 0 && (Buffer != RAM_BUFFER)) DosMark = *((ULONG *)(p+SE_Data+TD_SECTOR));
		EncodeSector(p+SE_Data, &EBN);
		EncodeLong(CheckSum(p+SE_Header, (TD_LABELSIZE+4)/2), p+SE_SumL);
		EncodeLong(CheckSum(p+SE_Data, (TD_SECTOR)/2), p+SE_SumD);
	}
	MFMTrack(Buf, &EBN);
	DMACheck[Track%8] = CheckTrack(Buf, SPT*BLOCKSIZE);
	SetBufferBorders(Buf);
	return(0);
}

/* We create a brand new track */

void CreateTrack(unsigned short Track, unsigned char *Buf, unsigned char UseFFS, unsigned char IntlFS, unsigned char DirCache) {

	register unsigned char i;
	register char *p;

	if (Track>159) return;

	for(i=0; i<SPT; i++) {
		p = Buf+i*BLOCKSIZE;
		SetBufferStart(p);
		EncodeLong(0xFF000000UL | (ULONG)Track<<16 | i<<8 | SPT-i, p+SE_Header);
		setmem(p+SE_Label, TD_LABELSIZE*2, 0xAA);
		MakeFormatData(Track*SPT+i, p+SE_Data+TD_SECTOR, UseFFS, IntlFS, DirCache);
		EncodeSector(p+SE_Data, &EBN);
		EncodeLong(CheckSum(p+SE_Header, (TD_LABELSIZE+4)/2), p+SE_SumL);
		EncodeLong(CheckSum(p+SE_Data, TD_SECTOR/2), p+SE_SumD);
	}
	MFMTrack(Buf, &EBN);
	DMACheck[Track%8] = CheckTrack(Buf, SPT*BLOCKSIZE);
	SetBufferBorders(Buf);
}


/* We take an MFM track just read and recode it for writing. */

signed char RecodeTrack(unsigned char *Buf, unsigned char Unit, unsigned short Track) {

	unsigned char SecNum;
	signed char Result = 0;

	SetBufferStart(Buf);
	SecNum = SPT-(unsigned char)(DecodeLong(Buf+SE_Header));
	if ((unsigned char)(DecodeLong(Buf+SE_Header)>>16) != Track) {
		PrintError(ReadID, Unit, Track, WrongTrackNumber);
		Result = 1;
	}

	if (SecNum>=SPT) {
		PrintError(ReadID, Unit, Track, WrongNextNumber);
		Result = 1;
	}
	else if (SecNum && ArrangeTrack(Buf+(SPT-SecNum)*BLOCKSIZE, GapSize, SecNum, &EBN)) {
		PrintError(ReadID, Unit, Track, NoFirstSector);
		Result  = 1;
	}
	else if (CheckCheckSums(Buf, SPT)) {
		PrintError(ReadID, Unit, Track, BadChecksum);
		Result = 1;
	}

	RenumberTrack(Buf);
	for(SecNum=0; SecNum<SPT; SecNum++) *((ULONG *)(Buf+SecNum*BLOCKSIZE)) = MFMZeroes;
	MFMTrack(Buf, &EBN);
	DMACheck[Track%8] = CheckTrack(Buf, SPT*BLOCKSIZE);
	SetBufferBorders(Buf);
	return(Result);
}


/* We verify a track. We return 1 on error. */

signed char VerifyTrack(unsigned char *Source, unsigned char *Verify, unsigned char Unit, unsigned short Track) {

	char VStart, *AfterGap;

	Verify += FindOffset(Verify);
	VStart = (char)(DecodeLong(Verify+SE_Header));
	if (VStart<1 || VStart>SPT) {
		PrintError(VerifyID, Unit, Track, WrongNextNumber);
		return(1);
	}
	*((ULONG *)Verify) = *((ULONG *)(Source+(SPT-VStart)*BLOCKSIZE));
	*((ULONG *)Verify+1) = *((ULONG *)(Source+(SPT-VStart)*BLOCKSIZE)+1);

	if (VStart != SPT) {
		if ((AfterGap = FindSync(Verify+VStart*BLOCKSIZE, GapSize)) == NULL) {
			PrintError(VerifyID, Unit, Track, NoFirstSector);
			return(1);
		}
		else {
			AfterGap += FindOffset(AfterGap);
			*((ULONG *)AfterGap) = *((ULONG *)Source);
			*((ULONG *)AfterGap+1) = *((ULONG *)Source+1);
		}
	}

	if (CompareTracks(Source, AfterGap, Verify, SPT-VStart, &EBN)) {
		PrintError(VerifyID, Unit, Track, VerifyError);
		return(1);
	}

	return(0);
}

/* We update the RootBlock of an MFM track */

void UpdateMFMRootBlock(unsigned char *Buf, unsigned char Date, unsigned char IncName) {

	register unsigned char FirstSector, *RootBlock = Buf;

	FirstSector = (unsigned char)(DecodeLong(Buf+SE_Header)>>8);
	if (FirstSector>=SPT) return;
	if (FirstSector != 0) RootBlock += (SPT-FirstSector)*BLOCKSIZE;
	DecodeSector(RootBlock+SE_Data, &EBN);
	UpdateRootBlock(RootBlock+SE_Data+TD_SECTOR, Date, IncName);
	EncodeSector(RootBlock+SE_Data, &EBN);
	EncodeLong(CheckSum(RootBlock+SE_Data, TD_SECTOR/2), RootBlock+SE_SumD);
	MFMTrack(Buf, &EBN);
	DMACheck[80%8] = CheckTrack(Buf, SPT*BLOCKSIZE);
}


/* From here onwards we aren't using the blitter, so we can call from anywhere. */

/* Standard or MFM or buffer DOS block recognition */

static unsigned char IsDosMark(ULONG Word) {
	if ((Word & 0xFFFFFF00UL) == MAKE_ID('D','O','S',0) && (Word & 0xFFUL)<6)
		return(1);
	else return(0);
}


unsigned char IsDosMFM(unsigned char *Buf) {

	register unsigned char FirstSector;

	FirstSector = (unsigned char)(DecodeLong(Buf+SE_Header)>>8);
	if (FirstSector>=SPT) return(0);

	if (FirstSector != 0) Buf += (SPT-FirstSector)*BLOCKSIZE;
	return(IsDosMark((*((ULONG *)(Buf+SE_Data)) & MFMMask)<<1 | (*((ULONG *)(Buf+SE_Data+TD_SECTOR)) & MFMMask)));
}

unsigned char IsDosBuffered(void) {
	return(IsDosMark(DosMark));
}


/* MFM or buffered disk name getting */

unsigned char *GetNameMFM(unsigned char *Buf) {

	register unsigned char i;

	i = (char)(DecodeLong(Buf+SE_Header)>>8);
	if (i>=SPT) strcpy(Name, ErrorName);
	else {
		if (i != 0) Buf += (SPT-i)*BLOCKSIZE;
		Buf += SE_Data+432;
		for(i=1; i<31; i++) Name[i-1] = (char)(((((ULONG *)Buf)[i/4] & MFMMask)<<1 | (((ULONG *)(Buf+TD_SECTOR))[i/4] & MFMMask))>>(24-(i%4)*8));
		i = (unsigned char)((*Buf & 0x55)<<1 | (*(Buf+TD_SECTOR) & 0x55));
		if (i>30) strcpy(Name, ErrorName);
		else Name[i] = 0;
	}
	return(Name);
}

unsigned char *GetNameBuffered(void) {
	return(Name);
}
